<?php

namespace App\Services;

use App\Models\Backup;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Str;
use ZipArchive;

class BackupService
{
    protected $backupDisk;
    protected $tempPath;

    public function __construct()
    {
        $this->tempPath = storage_path('app/temp');
        
        // Ensure temp directory exists
        if (!File::exists($this->tempPath)) {
            File::makeDirectory($this->tempPath, 0755, true);
        }
    }

    /**
     * Get backup disk instance (lazy loading)
     */
    protected function getBackupDisk()
    {
        if (!$this->backupDisk) {
            // Ensure backup disk is configured
            $this->ensureBackupDiskExists();
            $this->backupDisk = Storage::disk('backups');
        }
        return $this->backupDisk;
    }

    /**
     * Ensure backup disk configuration exists
     */
    protected function ensureBackupDiskExists()
    {
        if (!config('filesystems.disks.backups')) {
            $backupPath = storage_path('app/backups');
            if (!file_exists($backupPath)) {
                mkdir($backupPath, 0755, true);
            }

            config([
                'filesystems.disks.backups' => [
                    'driver' => 'local',
                    'root' => $backupPath,
                    'url' => env('APP_URL').'/storage/backups',
                    'visibility' => 'private',
                ]
            ]);
        }
    }

    /**
     * Create a new backup
     */
    public function createBackup($type, $description = null, $userId = null)
    {
        $backup = Backup::create([
            'name' => $this->generateBackupName($type),
            'original_name' => $this->generateOriginalName($type),
            'type' => $type,
            'status' => 'pending',
            'progress' => 0,
            'description' => $description,
            'created_by' => $userId
        ]);

        // Process backup in background (you might want to use queues for this)
        $this->processBackup($backup);

        return $backup;
    }

    /**
     * Process backup creation
     */
    protected function processBackup(Backup $backup)
    {
        try {
            $backup->update(['status' => 'processing', 'progress' => 10]);

            switch ($backup->type) {
                case 'database':
                    $this->createDatabaseBackup($backup);
                    break;
                case 'files':
                    $this->createFilesBackup($backup);
                    break;
                case 'full':
                    $this->createFullBackup($backup);
                    break;
            }

        } catch (\Exception $e) {
            $backup->markAsFailed($e->getMessage());
            throw $e;
        }
    }

    /**
     * Create database backup
     */
    protected function createDatabaseBackup(Backup $backup)
    {
        $backup->updateProgress(20, 'Starting database backup...');

        $filename = $backup->name . '.sql';
        $tempFile = $this->tempPath . '/' . $filename;

        // Get database configuration
        $database = config('database.connections.mysql.database');
        $username = config('database.connections.mysql.username');
        $password = config('database.connections.mysql.password');
        $host = config('database.connections.mysql.host');
        $port = config('database.connections.mysql.port', 3306);

        $backup->updateProgress(40, 'Exporting database...');

        // Create mysqldump command
        $command = sprintf(
            'mysqldump --user=%s --password=%s --host=%s --port=%s --single-transaction --routines --triggers %s > %s',
            escapeshellarg($username),
            escapeshellarg($password),
            escapeshellarg($host),
            escapeshellarg($port),
            escapeshellarg($database),
            escapeshellarg($tempFile)
        );

        exec($command, $output, $returnCode);

        if ($returnCode !== 0) {
            throw new \Exception('Database backup failed: ' . implode("\n", $output));
        }

        $backup->updateProgress(70, 'Compressing backup...');

        // Compress the SQL file
        $zipFile = $this->tempPath . '/' . $backup->name . '.zip';
        $this->createZipFile($zipFile, [$tempFile => $filename]);

        $backup->updateProgress(90, 'Storing backup...');

        // Move to backup storage
        $finalPath = 'database/' . date('Y/m/') . $backup->name . '.zip';
        $this->getBackupDisk()->put($finalPath, File::get($zipFile));

        // Calculate file size and checksum
        $fileSize = $this->getBackupDisk()->size($finalPath);
        $checksum = md5_file($zipFile);

        // Update backup record
        $backup->update([
            'file_path' => $finalPath,
            'file_size' => $fileSize,
            'checksum' => $checksum
        ]);

        // Cleanup temp files
        File::delete([$tempFile, $zipFile]);

        $backup->markAsCompleted($fileSize, $checksum);
    }

    /**
     * Create files backup
     */
    protected function createFilesBackup(Backup $backup)
    {
        $backup->updateProgress(20, 'Starting files backup...');

        $zipFile = $this->tempPath . '/' . $backup->name . '.zip';
        $zip = new ZipArchive();

        if ($zip->open($zipFile, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== TRUE) {
            throw new \Exception('Cannot create zip file');
        }

        $backup->updateProgress(30, 'Adding application files...');

        // Add important directories
        $directories = [
            'app' => base_path('app'),
            'config' => base_path('config'),
            'database' => base_path('database'),
            'resources' => base_path('resources'),
            'routes' => base_path('routes'),
            'public' => base_path('public'),
            'storage/app' => storage_path('app')
        ];

        $totalFiles = 0;
        $processedFiles = 0;

        // Count total files first
        foreach ($directories as $name => $path) {
            if (File::exists($path)) {
                $totalFiles += $this->countFiles($path);
            }
        }

        // Add files to zip
        foreach ($directories as $name => $path) {
            if (File::exists($path)) {
                $this->addDirectoryToZip($zip, $path, $name, $backup, $processedFiles, $totalFiles);
            }
        }

        $zip->close();

        $backup->updateProgress(90, 'Storing backup...');

        // Move to backup storage
        $finalPath = 'files/' . date('Y/m/') . $backup->name . '.zip';
        $this->getBackupDisk()->put($finalPath, File::get($zipFile));

        $fileSize = $this->getBackupDisk()->size($finalPath);
        $checksum = md5_file($zipFile);

        $backup->update([
            'file_path' => $finalPath,
            'file_size' => $fileSize,
            'checksum' => $checksum
        ]);

        File::delete($zipFile);
        $backup->markAsCompleted($fileSize, $checksum);
    }

    /**
     * Create full backup (database + files)
     */
    protected function createFullBackup(Backup $backup)
    {
        $backup->updateProgress(10, 'Starting full backup...');

        // Create temporary backups
        $dbBackup = Backup::create([
            'name' => $backup->name . '_db_temp',
            'type' => 'database',
            'status' => 'processing',
            'created_by' => $backup->created_by
        ]);

        $filesBackup = Backup::create([
            'name' => $backup->name . '_files_temp',
            'type' => 'files', 
            'status' => 'processing',
            'created_by' => $backup->created_by
        ]);

        // Create database backup
        $backup->updateProgress(30, 'Creating database backup...');
        $this->createDatabaseBackup($dbBackup);

        // Create files backup
        $backup->updateProgress(60, 'Creating files backup...');
        $this->createFilesBackup($filesBackup);

        // Combine both backups
        $backup->updateProgress(80, 'Combining backups...');
        
        $finalZip = $this->tempPath . '/' . $backup->name . '.zip';
        $zip = new ZipArchive();
        
        if ($zip->open($finalZip, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== TRUE) {
            throw new \Exception('Cannot create final zip file');
        }

        // Add database backup
        if ($this->getBackupDisk()->exists($dbBackup->file_path)) {
            $zip->addFromString('database.zip', $this->getBackupDisk()->get($dbBackup->file_path));
        }

        // Add files backup
        if ($this->getBackupDisk()->exists($filesBackup->file_path)) {
            $zip->addFromString('files.zip', $this->getBackupDisk()->get($filesBackup->file_path));
        }

        $zip->close();

        $backup->updateProgress(95, 'Storing full backup...');

        // Store final backup
        $finalPath = 'full/' . date('Y/m/') . $backup->name . '.zip';
        $this->getBackupDisk()->put($finalPath, File::get($finalZip));

        $fileSize = $this->getBackupDisk()->size($finalPath);
        $checksum = md5_file($finalZip);

        $backup->update([
            'file_path' => $finalPath,
            'file_size' => $fileSize,
            'checksum' => $checksum
        ]);

        // Cleanup temporary backups
        $this->getBackupDisk()->delete($dbBackup->file_path);
        $this->getBackupDisk()->delete($filesBackup->file_path);
        $dbBackup->delete();
        $filesBackup->delete();
        File::delete($finalZip);

        $backup->markAsCompleted($fileSize, $checksum);
    }

    /**
     * Restore database from backup
     */
    public function restoreBackup(Backup $backup)
    {
        if ($backup->type !== 'database') {
            throw new \Exception('Only database backups can be restored');
        }

        if (!$this->getBackupDisk()->exists($backup->file_path)) {
            throw new \Exception('Backup file not found');
        }

        // Download and extract backup
        $tempZip = $this->tempPath . '/restore_' . $backup->name . '.zip';
        $tempSql = $this->tempPath . '/restore_' . $backup->name . '.sql';

        File::put($tempZip, $this->getBackupDisk()->get($backup->file_path));

        // Extract SQL file
        $zip = new ZipArchive();
        if ($zip->open($tempZip) === TRUE) {
            $zip->extractTo($this->tempPath);
            $zip->close();
        }

        // Get database configuration
        $database = config('database.connections.mysql.database');
        $username = config('database.connections.mysql.username');
        $password = config('database.connections.mysql.password');
        $host = config('database.connections.mysql.host');
        $port = config('database.connections.mysql.port', 3306);

        // Restore database
        $command = sprintf(
            'mysql --user=%s --password=%s --host=%s --port=%s %s < %s',
            escapeshellarg($username),
            escapeshellarg($password),
            escapeshellarg($host),
            escapeshellarg($port),
            escapeshellarg($database),
            escapeshellarg($tempSql)
        );

        exec($command, $output, $returnCode);

        // Cleanup
        File::delete([$tempZip, $tempSql]);

        if ($returnCode !== 0) {
            throw new \Exception('Database restore failed: ' . implode("\n", $output));
        }
    }

    /**
     * Get disk space information
     */
    public function getDiskSpaceInfo()
    {
        $backupPath = config('filesystems.disks.backups.root');
        
        return [
            'total' => disk_total_space($backupPath),
            'free' => disk_free_space($backupPath),
            'used' => disk_total_space($backupPath) - disk_free_space($backupPath)
        ];
    }

    /**
     * Get backup statistics
     */
    public function getBackupStats()
    {
        return [
            'total_backups' => Backup::count(),
            'completed_backups' => Backup::where('status', 'completed')->count(),
            'failed_backups' => Backup::where('status', 'failed')->count(),
            'total_size' => Backup::where('status', 'completed')->sum('file_size'),
            'database_backups' => Backup::where('type', 'database')->where('status', 'completed')->count(),
            'files_backups' => Backup::where('type', 'files')->where('status', 'completed')->count(),
            'full_backups' => Backup::where('type', 'full')->where('status', 'completed')->count(),
        ];
    }

    /**
     * Cleanup old backups
     */
    public function cleanupOldBackups($days)
    {
        $cutoffDate = now()->subDays($days);
        $oldBackups = Backup::where('created_at', '<', $cutoffDate)->get();
        
        $deletedCount = 0;
        foreach ($oldBackups as $backup) {
            if ($this->getBackupDisk()->exists($backup->file_path)) {
                $this->getBackupDisk()->delete($backup->file_path);
            }
            $backup->delete();
            $deletedCount++;
        }

        return $deletedCount;
    }

    // Helper methods
    protected function generateBackupName($type)
    {
        return sprintf(
            '%s_%s_%s',
            $type,
            date('Y-m-d_H-i-s'),
            Str::random(8)
        );
    }

    protected function generateOriginalName($type)
    {
        return sprintf(
            '%s_backup_%s.zip',
            $type,
            date('Y-m-d_H-i-s')
        );
    }

    protected function createZipFile($zipPath, $files)
    {
        $zip = new ZipArchive();
        if ($zip->open($zipPath, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== TRUE) {
            throw new \Exception('Cannot create zip file');
        }

        foreach ($files as $filePath => $archiveName) {
            $zip->addFile($filePath, $archiveName);
        }

        $zip->close();
    }

    protected function addDirectoryToZip($zip, $path, $archivePath, $backup, &$processedFiles, $totalFiles)
    {
        $iterator = new \RecursiveIteratorIterator(
            new \RecursiveDirectoryIterator($path),
            \RecursiveIteratorIterator::SELF_FIRST
        );

        foreach ($iterator as $file) {
            if ($file->isFile()) {
                $filePath = $file->getRealPath();
                $relativePath = $archivePath . '/' . substr($filePath, strlen($path) + 1);
                
                $zip->addFile($filePath, $relativePath);
                $processedFiles++;

                // Update progress every 100 files
                if ($processedFiles % 100 === 0) {
                    $progress = 30 + (($processedFiles / $totalFiles) * 50);
                    $backup->updateProgress(min(80, $progress), "Processing files... ({$processedFiles}/{$totalFiles})");
                }
            }
        }
    }

    protected function countFiles($path)
    {
        $count = 0;
        $iterator = new \RecursiveIteratorIterator(
            new \RecursiveDirectoryIterator($path),
            \RecursiveIteratorIterator::SELF_FIRST
        );

        foreach ($iterator as $file) {
            if ($file->isFile()) {
                $count++;
            }
        }

        return $count;
    }
}