<?php

namespace App\Services;

use App\Models\Notification;
use App\Models\User;
use Illuminate\Support\Facades\Log;

class NotificationService
{
    /**
     * Bildirim türleri
     */
    const TYPE_TASK_CREATED = 'task_created';
    const TYPE_TASK_APPROVED = 'task_approved';
    const TYPE_TASK_REJECTED = 'task_rejected';
    const TYPE_TICKET_CREATED = 'ticket_created';
    const TYPE_TICKET_REPLIED = 'ticket_replied';
    const TYPE_ANNOUNCEMENT = 'announcement';
    const TYPE_WITHDRAWAL_REQUEST = 'withdrawal_request';
    const TYPE_WITHDRAWAL_APPROVED = 'withdrawal_approved';
    const TYPE_WITHDRAWAL_REJECTED = 'withdrawal_rejected';
    const TYPE_BALANCE_REQUEST = 'balance_request';
    const TYPE_BALANCE_APPROVED = 'balance_approved';
    const TYPE_BALANCE_REJECTED = 'balance_rejected';
    const TYPE_TASK_PROOF_APPROVED = 'task_proof_approved';
    const TYPE_TASK_PROOF_REJECTED = 'task_proof_rejected';

    /**
     * Yeni görev oluşturulduğunda bildirim gönder
     */
    public function notifyTaskCreated($task, $creator)
    {
        try {
            // Mevcut locale'i kaydet
            $currentLocale = app()->getLocale();
            
            // Admin kullanıcılara bildirim gönder
            $admins = User::where('user_type', 1)->get();
            
            foreach ($admins as $admin) {
                // Admin'in dil ayarını kullan (varsa), yoksa Türkçe varsayılan
                app()->setLocale('tr');
                
                $this->createNotification([
                    'user_id' => $admin->id,
                    'type' => self::TYPE_TASK_CREATED,
                    'title' => __('New Task Created'),
                    'description' => __('A new task ":task_name" has been created by :user_name with reward :reward₺', [
                        'task_name' => $task->name,
                        'user_name' => $creator->account_holder,
                        'reward' => number_format($task->reward, 2)
                    ]),
                    'link' => '/admin/tasks/' . $task->id,
                    'expires_at' => now()->addDays(7)
                ]);
            }

            // Görev sahibine onay bekliyor bildirimi
            app()->setLocale('tr');
            $this->createNotification([
                'user_id' => $creator->id,
                'type' => self::TYPE_TASK_CREATED,
                'title' => __('Task Created Successfully'),
                'description' => __('Your task ":task_name" has been created and is waiting for admin approval.', [
                    'task_name' => $task->name
                ]),
                'link' => '/tasks/' . $task->id,
                'expires_at' => now()->addDays(7)
            ]);
            
            // Locale'i geri yükle
            app()->setLocale($currentLocale);

        } catch (\Exception $e) {
            Log::error('Task creation notification error: ' . $e->getMessage());
        }
    }

    /**
     * Görev onaylandığında bildirim gönder
     */
    public function notifyTaskApproved($task)
    {
        try {
            $this->createNotification([
                'user_id' => $task->user_id,
                'type' => self::TYPE_TASK_APPROVED,
                'title' => __('Task Approved'),
                'description' => __('Your task ":task_name" has been approved and is now active!', [
                    'task_name' => $task->name
                ]),
                'link' => '/tasks/' . $task->id,
                'expires_at' => now()->addDays(7)
            ]);

            // Tüm kullanıcılara yeni görev bildirimi (opsiyonel)
            $users = User::where('user_type', 2)->where('id', '!=', $task->user_id)->get();
            foreach ($users as $user) {
                $this->createNotification([
                    'user_id' => $user->id,
                    'type' => 'new_content',
                    'title' => __('New Task Available'),
                    'description' => $task->reward,
                    'link' => '/tasks/' . $task->id,
                    'expires_at' => now()->addDays(3)
                ]);
            }

        } catch (\Exception $e) {
            Log::error('Task approval notification error: ' . $e->getMessage());
        }
    }

    /**
     * Ticket oluşturulduğunda bildirim gönder
     */
    public function notifyTicketCreated($ticket)
    {
        try {
            // Admin kullanıcılara bildirim gönder
            $admins = User::where('user_type', 1)->get();
            
            foreach ($admins as $admin) {
                $this->createNotification([
                    'user_id' => $admin->id,
                    'type' => self::TYPE_TICKET_CREATED,
                    'title' => __('New Support Ticket'),
                    'description' => __('New ticket #:ticket_id created by :user_name: ":subject"', [
                        'ticket_id' => $ticket->id,
                        'user_name' => $ticket->user->account_holder,
                        'subject' => $ticket->subject
                    ]),
                    'link' => '/admin/tickets/' . $ticket->id,
                    'expires_at' => now()->addDays(7)
                ]);
            }

        } catch (\Exception $e) {
            Log::error('Ticket creation notification error: ' . $e->getMessage());
        }
    }

    /**
     * Ticket'a cevap verildiğinde bildirim gönder
     */
    public function notifyTicketReplied($ticket, $replier)
    {
        try {
            // Ticket sahibine bildirim gönder (eğer cevaplayan ticket sahibi değilse)
            if ($ticket->user_id != $replier->id) {
                $this->createNotification([
                    'user_id' => $ticket->user_id,
                    'type' => self::TYPE_TICKET_REPLIED,
                    'title' => __('Ticket Reply Received'),
                    'description' => __('You have received a new reply for ticket #:ticket_id', [
                        'ticket_id' => $ticket->id
                    ]),
                    'link' => '/tickets/' . $ticket->id,
                    'expires_at' => now()->addDays(7)
                ]);
            }

            // Eğer kullanıcı cevap verdiyse admin'lere bildirim gönder
            if ($replier->user_type != 1) {
                $admins = User::where('user_type', 1)->get();
                foreach ($admins as $admin) {
                    $this->createNotification([
                        'user_id' => $admin->id,
                        'type' => self::TYPE_TICKET_REPLIED,
                        'title' => __('Ticket Reply from User'),
                        'description' => __('User replied to ticket #:ticket_id', [
                            'ticket_id' => $ticket->id
                        ]),
                        'link' => '/admin/tickets/' . $ticket->id,
                        'expires_at' => now()->addDays(7)
                    ]);
                }
            }

        } catch (\Exception $e) {
            Log::error('Ticket reply notification error: ' . $e->getMessage());
        }
    }

    /**
     * Duyuru eklendiğinde bildirim gönder
     */
    public function notifyAnnouncementCreated($announcement)
    {
        try {
            // Tüm kullanıcılara bildirim gönder
            $users = User::where('user_type', '!=', 1)->get(); // Admin hariç
            
            foreach ($users as $user) {
                $this->createNotification([
                    'user_id' => $user->id,
                    'type' => self::TYPE_ANNOUNCEMENT,
                    'title' => __('New Announcement'),
                    'description' => $announcement->title,
                    'link' => '/announcements/' . $announcement->id,
                    'expires_at' => now()->addDays(14)
                ]);
            }

        } catch (\Exception $e) {
            Log::error('Announcement notification error: ' . $e->getMessage());
        }
    }

    /**
     * Çekim talebi oluşturulduğunda bildirim gönder
     */
    public function notifyWithdrawalRequest($withdrawal)
    {
        try {
            // Admin kullanıcılara bildirim gönder
            $admins = User::where('user_type', 1)->get();
            
            foreach ($admins as $admin) {
                $this->createNotification([
                    'user_id' => $admin->id,
                    'type' => self::TYPE_WITHDRAWAL_REQUEST,
                    'title' => __('New Withdrawal Request'),
                    'description' => __('New withdrawal request of :amount₺ from :user_name', [
                        'amount' => number_format($withdrawal->amount, 2),
                        'user_name' => $withdrawal->user->account_holder
                    ]),
                    'link' => '/admin/withdrawals/' . $withdrawal->id,
                    'expires_at' => now()->addDays(7)
                ]);
            }

        } catch (\Exception $e) {
            Log::error('Withdrawal request notification error: ' . $e->getMessage());
        }
    }

    /**
     * Çekim talebi onaylandığında bildirim gönder
     */
    public function notifyWithdrawalApproved($withdrawal)
    {
        try {
            $this->createNotification([
                'user_id' => $withdrawal->user_id,
                'type' => self::TYPE_WITHDRAWAL_APPROVED,
                'title' => __('Withdrawal Approved'),
                'description' => __('Your withdrawal request of :amount₺ has been approved!', [
                    'amount' => number_format($withdrawal->amount, 2)
                ]),
                'link' => '/withdrawals',
                'expires_at' => now()->addDays(7)
            ]);

        } catch (\Exception $e) {
            Log::error('Withdrawal approval notification error: ' . $e->getMessage());
        }
    }

    /**
     * Bakiye talebi onaylandığında bildirim gönder
     */
    public function notifyBalanceApproved($balanceRequest)
    {
        try {
            $this->createNotification([
                'user_id' => $balanceRequest->user_id,
                'type' => self::TYPE_BALANCE_APPROVED,
                'title' => __('Balance Request Approved'),
                'description' => __('Your balance request of :amount₺ has been approved and added to your account!', [
                    'amount' => number_format($balanceRequest->amount, 2)
                ]),
                'link' => '/user/profile',
                'expires_at' => now()->addDays(7)
            ]);

        } catch (\Exception $e) {
            Log::error('Balance approval notification error: ' . $e->getMessage());
        }
    }

    /**
     * Görev kanıtı onaylandığında bildirim gönder
     */
    public function notifyTaskProofApproved($taskProof)
    {
        try {
            $this->createNotification([
                'user_id' => $taskProof->UserId,
                'type' => self::TYPE_TASK_PROOF_APPROVED,
                'title' => __('Task Proof Approved'),
                'description' => __('Your proof for task ":task_name" has been approved! Reward: :reward₺', [
                    'task_name' => $taskProof->task->name,
                    'reward' => number_format($taskProof->task->reward, 2)
                ]),
                'link' => '/tasks/' . $taskProof->TaskId,
                'expires_at' => now()->addDays(7)
            ]);

        } catch (\Exception $e) {
            Log::error('Task proof approval notification error: ' . $e->getMessage());
        }
    }

    /**
     * Görev kanıtı reddedildiğinde bildirim gönder
     */
    public function notifyTaskProofRejected($taskProof)
    {
        try {
            $this->createNotification([
                'user_id' => $taskProof->UserId,
                'type' => self::TYPE_TASK_PROOF_REJECTED,
                'title' => __('Task Proof Rejected'),
                'description' => __('Your proof for task ":task_name" has been rejected. Reason: :reason', [
                    'task_name' => $taskProof->task->name,
                    'reason' => $taskProof->Reason ?? __('No reason provided')
                ]),
                'link' => '/tasks/' . $taskProof->TaskId,
                'expires_at' => now()->addDays(7)
            ]);

        } catch (\Exception $e) {
            Log::error('Task proof rejection notification error: ' . $e->getMessage());
        }
    }

    /**
     * Bildirim oluştur
     */
    private function createNotification($data)
    {
        return Notification::create(array_merge([
            'status' => 1, // 1 = unread, 0 = read
            'read_at' => null,
        ], $data));
    }

    /**
     * Kullanıcının okunmamış bildirim sayısını getir
     */
    public function getUnreadCount($userId)
    {
        return Notification::where('user_id', $userId)
            ->where('status', 1) // 1 = unread
            ->where(function($query) {
                $query->whereNull('expires_at')
                      ->orWhere('expires_at', '>', now());
            })
            ->count();
    }

    /**
     * Kullanıcının bildirimlerini getir
     */
    public function getUserNotifications($userId, $limit = 10)
    {
        return Notification::where('user_id', $userId)
            ->where(function($query) {
                $query->whereNull('expires_at')
                      ->orWhere('expires_at', '>', now());
            })
            ->orderBy('created_at', 'desc')
            ->limit($limit)
            ->get();
    }

    /**
     * Süresi dolmuş bildirimleri temizle
     */
    public function cleanExpiredNotifications()
    {
        return Notification::where('expires_at', '<', now())->delete();
    }
}