@extends('layouts.app')

@section('title', __('Backup Management'))

@section('style')
<link href="{{ asset('assets/plugins/notifications/css/lobibox.min.css') }}" rel="stylesheet">
<link href="{{ asset('assets/plugins/datatable/css/dataTables.bootstrap5.min.css') }}" rel="stylesheet">
<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
<style>
    .backup-card {
        border: 1px solid #e9ecef;
        border-radius: 16px;
        transition: all 0.3s ease;
        background: white;
        box-shadow: 0 2px 8px rgba(0,0,0,0.08);
        margin-bottom: 20px;
    }
    
    .backup-card:hover {
        box-shadow: 0 8px 30px rgba(0,0,0,0.15);
        transform: translateY(-3px);
    }
    
    .backup-header {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        padding: 20px;
        border-radius: 16px 16px 0 0;
        position: relative;
    }
    
    .backup-header::after {
        content: '';
        position: absolute;
        bottom: 0;
        left: 0;
        right: 0;
        height: 4px;
        background: linear-gradient(90deg, #28a745, #20c997, #17a2b8);
    }
    
    .stats-card {
        background: white;
        border-radius: 12px;
        padding: 20px;
        box-shadow: 0 4px 15px rgba(0,0,0,0.1);
        border: none;
        transition: transform 0.3s ease;
    }
    
    .stats-card:hover {
        transform: translateY(-5px);
    }
    
    .stats-icon {
        width: 60px;
        height: 60px;
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 24px;
        color: white;
        margin-bottom: 15px;
    }
    
    .progress-custom {
        height: 8px;
        border-radius: 10px;
        background-color: #f8f9fa;
    }
    
    .progress-custom .progress-bar {
        border-radius: 10px;
    }
    
    .disk-usage {
        background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);
        color: white;
        border-radius: 12px;
        padding: 20px;
    }
    
    .action-buttons .btn {
        margin: 2px;
        border-radius: 8px;
        font-size: 0.85rem;
    }
    
    .backup-type-badge {
        font-size: 0.75rem;
        padding: 4px 8px;
        border-radius: 12px;
        font-weight: 600;
    }
    
    .file-size {
        font-family: 'Courier New', monospace;
        font-weight: bold;
    }
</style>
@endsection

@section('wrapper')
<div class="page-wrapper">
    <div class="page-content">
        <!-- Header -->
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-3">
            <div class="breadcrumb-title pe-3">{{__('Admin')}}</div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ url('/home') }}"><i class="bx bx-home-alt"></i></a></li>
                        <li class="breadcrumb-item active" aria-current="page">{{__('Backup Management')}}</li>
                    </ol>
                </nav>
            </div>
            <div class="ms-auto">
                <button class="btn btn-primary" onclick="showCreateBackupModal()">
                    <i class="bx bx-plus"></i> {{__('Create Backup')}}
                </button>
            </div>
        </div>

        <!-- Statistics Cards -->
        <div class="row mb-4">
            <div class="col-lg-3 col-md-6">
                <div class="stats-card">
                    <div class="stats-icon bg-primary mx-auto">
                        <i class="bx bx-data"></i>
                    </div>
                    <div class="text-center">
                        <h4 class="mb-1">{{ $stats['total_backups'] }}</h4>
                        <p class="text-muted mb-0">{{__('Total Backups')}}</p>
                    </div>
                </div>
            </div>
            <div class="col-lg-3 col-md-6">
                <div class="stats-card">
                    <div class="stats-icon bg-success mx-auto">
                        <i class="bx bx-check-circle"></i>
                    </div>
                    <div class="text-center">
                        <h4 class="mb-1">{{ $stats['completed_backups'] }}</h4>
                        <p class="text-muted mb-0">{{__('Completed')}}</p>
                    </div>
                </div>
            </div>
            <div class="col-lg-3 col-md-6">
                <div class="stats-card">
                    <div class="stats-icon bg-danger mx-auto">
                        <i class="bx bx-x-circle"></i>
                    </div>
                    <div class="text-center">
                        <h4 class="mb-1">{{ $stats['failed_backups'] }}</h4>
                        <p class="text-muted mb-0">{{__('Failed')}}</p>
                    </div>
                </div>
            </div>
            <div class="col-lg-3 col-md-6">
                <div class="stats-card">
                    <div class="stats-icon bg-info mx-auto">
                        <i class="bx bx-hdd"></i>
                    </div>
                    <div class="text-center">
                        <h4 class="mb-1 file-size">{{ \App\Helpers\BackupHelper::formatBytes($stats['total_size']) }}</h4>
                        <p class="text-muted mb-0">{{__('Total Size')}}</p>
                    </div>
                </div>
            </div>
        </div>

        <!-- Disk Usage -->
        <div class="row mb-4">
            <div class="col-12">
                <div class="disk-usage">
                    <div class="row align-items-center">
                        <div class="col-md-8">
                            <h5 class="mb-2">{{__('Disk Usage')}}</h5>
                            <div class="progress progress-custom mb-2">
                                @php
                                    $usedPercentage = $diskSpace['total'] > 0 ? ($diskSpace['used'] / $diskSpace['total']) * 100 : 0;
                                @endphp
                                <div class="progress-bar bg-white" style="width: {{ $usedPercentage }}%"></div>
                            </div>
                            <small>
                                {{__('Used')}}: {{ \App\Helpers\BackupHelper::formatBytes($diskSpace['used']) }} / 
                                {{__('Total')}}: {{ \App\Helpers\BackupHelper::formatBytes($diskSpace['total']) }} 
                                ({{ number_format($usedPercentage, 1) }}%)
                            </small>
                        </div>
                        <div class="col-md-4 text-end">
                            <button class="btn btn-outline-light" onclick="showCleanupModal()">
                                <i class="bx bx-trash"></i> {{__('Cleanup Old Backups')}}
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Backups List -->
        <div class="backup-card">
            <div class="backup-header">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <h4 class="mb-1">{{__('Backup History')}}</h4>
                        <p class="mb-0 opacity-75">{{__('Manage your system backups')}}</p>
                    </div>
                    <div>
                        <button class="btn btn-outline-light btn-sm" onclick="refreshBackups()">
                            <i class="bx bx-refresh"></i> {{__('Refresh')}}
                        </button>
                    </div>
                </div>
            </div>
            
            <div class="card-body">
                <div class="table-responsive">
                    <table id="backupsTable" class="table table-striped">
                        <thead>
                            <tr>
                                <th>{{__('Name')}}</th>
                                <th>{{__('Type')}}</th>
                                <th>{{__('Size')}}</th>
                                <th>{{__('Status')}}</th>
                                <th>{{__('Progress')}}</th>
                                <th>{{__('Created')}}</th>
                                <th>{{__('Created By')}}</th>
                                <th>{{__('Actions')}}</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($backups as $backup)
                            <tr id="backup-{{ $backup->id }}">
                                <td>
                                    <div>
                                        <strong>{{ $backup->original_name }}</strong>
                                        @if($backup->description)
                                            <br><small class="text-muted">{{ $backup->description }}</small>
                                        @endif
                                    </div>
                                </td>
                                <td>
                                    <span class="badge bg-{{ $backup->type_badge }} backup-type-badge">
                                        {{ $backup->type_text }}
                                    </span>
                                </td>
                                <td>
                                    <span class="file-size">
                                        {{ $backup->file_size ? \App\Helpers\BackupHelper::formatBytes($backup->file_size) : '-' }}
                                    </span>
                                </td>
                                <td>
                                    <span class="badge bg-{{ $backup->status_badge }}">
                                        {{ $backup->status_text }}
                                    </span>
                                </td>
                                <td>
                                    @if($backup->status === 'processing')
                                        <div class="progress progress-custom" style="height: 6px;">
                                            <div class="progress-bar bg-info" style="width: {{ $backup->progress }}%"></div>
                                        </div>
                                        <small>{{ $backup->progress }}%</small>
                                    @else
                                        <span class="text-muted">-</span>
                                    @endif
                                </td>
                                <td>
                                    <div>
                                        {{ $backup->created_at->format('d.m.Y H:i') }}
                                        <br><small class="text-muted">{{ $backup->age }}</small>
                                    </div>
                                </td>
                                <td>
                                    {{ $backup->creator->account_holder ?? __('System') }}
                                </td>
                                <td>
                                    <div class="action-buttons">
                                        @if($backup->canBeDownloaded())
                                            <a href="{{ route('admin.backups.download', $backup->id) }}" 
                                               class="btn btn-sm btn-outline-primary" 
                                               title="{{__('Download')}}">
                                                <i class="bx bx-download"></i>
                                            </a>
                                        @endif
                                        
                                        @if($backup->canBeRestored())
                                            <button class="btn btn-sm btn-outline-warning" 
                                                    onclick="showRestoreModal({{ $backup->id }})"
                                                    title="{{__('Restore')}}">
                                                <i class="bx bx-reset"></i>
                                            </button>
                                        @endif
                                        
                                        <button class="btn btn-sm btn-outline-danger" 
                                                onclick="deleteBackup({{ $backup->id }})"
                                                title="{{__('Delete')}}">
                                            <i class="bx bx-trash"></i>
                                        </button>
                                    </div>
                                </td>
                            </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>
                
                <!-- Pagination -->
                <div class="d-flex justify-content-center mt-3">
                    {{ $backups->links() }}
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Create Backup Modal -->
<div class="modal fade" id="createBackupModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">{{__('Create New Backup')}}</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form id="createBackupForm">
                <div class="modal-body">
                    <div class="mb-3">
                        <label class="form-label">{{__('Backup Type')}} <span class="text-danger">*</span></label>
                        <select class="form-select" name="type" required>
                            <option value="">{{__('Select backup type')}}</option>
                            <option value="database">{{__('Database Only')}}</option>
                            <option value="files">{{__('Files Only')}}</option>
                            <option value="full">{{__('Full Backup (Database + Files)')}}</option>
                        </select>
                        <div class="form-text">
                            <small>
                                <strong>{{__('Database Only')}}:</strong> {{__('Exports all database tables and data')}}<br>
                                <strong>{{__('Files Only')}}:</strong> {{__('Backs up application files and uploads')}}<br>
                                <strong>{{__('Full Backup')}}:</strong> {{__('Complete backup including database and files')}}
                            </small>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">{{__('Description')}} <span class="text-muted">({{__('Optional')}})</span></label>
                        <textarea class="form-control" name="description" rows="3" 
                                  placeholder="{{__('Add a description for this backup...')}}"></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">{{__('Cancel')}}</button>
                    <button type="submit" class="btn btn-primary">
                        <i class="bx bx-plus"></i> {{__('Create Backup')}}
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Restore Modal -->
<div class="modal fade" id="restoreModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">{{__('Restore Database')}}</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form id="restoreForm">
                <div class="modal-body">
                    <div class="alert alert-warning">
                        <i class="bx bx-warning"></i>
                        <strong>{{__('Warning!')}}</strong> 
                        {{__('This will replace all current database data with the backup data. This action cannot be undone.')}}
                    </div>
                    
                    <div class="form-check">
                        <input class="form-check-input" type="checkbox" name="confirm" id="confirmRestore" required>
                        <label class="form-check-label" for="confirmRestore">
                            {{__('I understand that this will overwrite all current data')}}
                        </label>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">{{__('Cancel')}}</button>
                    <button type="submit" class="btn btn-danger">
                        <i class="bx bx-reset"></i> {{__('Restore Database')}}
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Cleanup Modal -->
<div class="modal fade" id="cleanupModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">{{__('Cleanup Old Backups')}}</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form id="cleanupForm">
                <div class="modal-body">
                    <div class="mb-3">
                        <label class="form-label">{{__('Delete backups older than')}} <span class="text-danger">*</span></label>
                        <div class="input-group">
                            <input type="number" class="form-control" name="days" min="1" max="365" value="30" required>
                            <span class="input-group-text">{{__('days')}}</span>
                        </div>
                        <div class="form-text">{{__('Backups older than this many days will be permanently deleted')}}</div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">{{__('Cancel')}}</button>
                    <button type="submit" class="btn btn-warning">
                        <i class="bx bx-trash"></i> {{__('Cleanup Backups')}}
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>
@endsection

@section('script')
<script src="{{ asset('assets/plugins/notifications/js/lobibox.min.js') }}"></script>
<script src="{{ asset('assets/plugins/datatable/js/jquery.dataTables.min.js') }}"></script>
<script src="{{ asset('assets/plugins/datatable/js/dataTables.bootstrap5.min.js') }}"></script>

<script>
let currentRestoreId = null;

$(document).ready(function() {
    // Initialize DataTable
    $('#backupsTable').DataTable({
        pageLength: 25,
        responsive: true,
        order: [[5, 'desc']], // Order by created date desc
        language: {
            processing: "{{__('Processing...')}}",
            search: "{{__('Search:')}}",
            lengthMenu: "{{__('Show _MENU_ entries')}}",
            info: "{{__('Showing _START_ to _END_ of _TOTAL_ entries')}}",
            infoEmpty: "{{__('Showing 0 to 0 of 0 entries')}}",
            infoFiltered: "{{__('(filtered from _MAX_ total entries)')}}",
            paginate: {
                first: "{{__('First')}}",
                last: "{{__('Last')}}",
                next: "{{__('Next')}}",
                previous: "{{__('Previous')}}"
            }
        }
    });

    // Auto-refresh processing backups every 5 seconds
    setInterval(function() {
        refreshProcessingBackups();
    }, 5000);
});

// Show create backup modal
function showCreateBackupModal() {
    $('#createBackupModal').modal('show');
}

// Show restore modal
function showRestoreModal(backupId) {
    currentRestoreId = backupId;
    $('#restoreModal').modal('show');
}

// Show cleanup modal
function showCleanupModal() {
    $('#cleanupModal').modal('show');
}

// Create backup form submission
$('#createBackupForm').on('submit', function(e) {
    e.preventDefault();
    
    const formData = new FormData(this);
    
    $.ajax({
        url: '{{ route("admin.backups.create") }}',
        method: 'POST',
        data: formData,
        processData: false,
        contentType: false,
        headers: {
            'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
        },
        success: function(response) {
            if (response.success) {
                showNotification('success', response.message);
                $('#createBackupModal').modal('hide');
                setTimeout(() => location.reload(), 1500);
            } else {
                showNotification('error', response.message);
            }
        },
        error: function(xhr) {
            const message = xhr.responseJSON?.message || '{{__("Error creating backup")}}';
            showNotification('error', message);
        }
    });
});

// Restore form submission
$('#restoreForm').on('submit', function(e) {
    e.preventDefault();
    
    if (!currentRestoreId) return;
    
    const formData = new FormData(this);
    
    $.ajax({
        url: `/admin/backups/${currentRestoreId}/restore`,
        method: 'POST',
        data: formData,
        processData: false,
        contentType: false,
        headers: {
            'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
        },
        success: function(response) {
            if (response.success) {
                showNotification('success', response.message);
                $('#restoreModal').modal('hide');
            } else {
                showNotification('error', response.message);
            }
        },
        error: function(xhr) {
            const message = xhr.responseJSON?.message || '{{__("Error restoring backup")}}';
            showNotification('error', message);
        }
    });
});

// Cleanup form submission
$('#cleanupForm').on('submit', function(e) {
    e.preventDefault();
    
    const formData = new FormData(this);
    
    $.ajax({
        url: '{{ route("admin.backups.cleanup") }}',
        method: 'POST',
        data: formData,
        processData: false,
        contentType: false,
        headers: {
            'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
        },
        success: function(response) {
            if (response.success) {
                showNotification('success', response.message);
                $('#cleanupModal').modal('hide');
                setTimeout(() => location.reload(), 1500);
            } else {
                showNotification('error', response.message);
            }
        },
        error: function(xhr) {
            const message = xhr.responseJSON?.message || '{{__("Error during cleanup")}}';
            showNotification('error', message);
        }
    });
});

// Delete backup
function deleteBackup(backupId) {
    Swal.fire({
        title: '{{__("Are you sure?")}}',
        text: '{{__("This backup will be permanently deleted!")}}',
        icon: 'warning',
        showCancelButton: true,
        confirmButtonText: '{{__("Yes, delete it!")}}',
        cancelButtonText: '{{__("Cancel")}}',
        reverseButtons: true
    }).then((result) => {
        if (result.isConfirmed) {
            $.ajax({
                url: `/admin/backups/${backupId}`,
                method: 'POST',
                data: { _method: 'DELETE' },
                headers: {
                    'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
                },
                success: function(response) {
                    if (response.success) {
                        showNotification('success', response.message);
                        $(`#backup-${backupId}`).fadeOut();
                    } else {
                        showNotification('error', response.message);
                    }
                },
                error: function(xhr) {
                    const message = xhr.responseJSON?.message || '{{__("Error deleting backup")}}';
                    showNotification('error', message);
                }
            });
        }
    });
}

// Refresh backups
function refreshBackups() {
    location.reload();
}

// Refresh processing backups
function refreshProcessingBackups() {
    $('tr[id^="backup-"]').each(function() {
        const row = $(this);
        const backupId = row.attr('id').replace('backup-', '');
        const statusBadge = row.find('.badge');
        
        if (statusBadge.text().trim() === '{{__("Processing")}}') {
            $.ajax({
                url: `/admin/backups/${backupId}/progress`,
                method: 'GET',
                success: function(response) {
                    if (response.status === 'completed') {
                        location.reload();
                    } else if (response.status === 'failed') {
                        location.reload();
                    } else {
                        // Update progress
                        const progressBar = row.find('.progress-bar');
                        const progressText = row.find('small');
                        
                        if (progressBar.length) {
                            progressBar.css('width', response.progress + '%');
                            progressText.text(response.progress + '%');
                        }
                    }
                }
            });
        }
    });
}

// Show notification
function showNotification(type, message) {
    Lobibox.notify(type, {
        pauseDelayOnHover: true,
        continueDelayOnInactiveTab: false,
        position: 'top right',
        icon: type === 'success' ? 'bx bx-check-circle' : 'bx bx-x-circle',
        msg: message,
        delay: 4000
    });
}

</script>
@endsection