@extends('layouts.app')

@section('title', __('Balance Requests'))

@section('wrapper')
<div class="page-wrapper">
    <div class="page-content">
        <!-- Breadcrumb -->
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-3">
            <div class="breadcrumb-title pe-3">{{ __('Admin Panel') }}</div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ route('dashboard') }}"><i class="bx bx-home-alt"></i></a></li>
                        <li class="breadcrumb-item active" aria-current="page">{{ __('Balance Requests') }}</li>
                    </ol>
                </nav>
            </div>
        </div>

        <!-- Header Section -->
        <div class="row">
            <div class="col-12">
                <div class="card border-0 shadow-sm">
                    <div class="card-header bg-gradient-primary text-white">
                        <div class="d-flex flex-column flex-md-row justify-content-between align-items-start align-items-md-center">
                            <div class="d-flex align-items-center mb-2 mb-md-0">
                                <i class="bx bx-wallet fs-4 me-2"></i>
                                <h5 class="mb-0 text-white">{{ __('Balance Requests') }}</h5>
                            </div>
                            <div class="d-flex flex-column flex-sm-row gap-2">
                                <!-- Status Filter -->
                                <select class="form-select form-select-sm" id="statusFilter" style="min-width: 150px;">
                                    <option value="">{{ __('All Statuses') }}</option>
                                    <option value="pending">{{ __('Pending') }}</option>
                                    <option value="completed">{{ __('Approved') }}</option>
                                    <option value="rejected">{{ __('Rejected') }}</option>
                                </select>
                                <!-- Refresh Button -->
                                <button class="btn btn-light btn-sm" onclick="refreshRequests()">
                                    <i class="bx bx-refresh"></i>
                                    <span class="d-none d-sm-inline ms-1">{{ __('Refresh') }}</span>
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Stats Cards -->
        <div class="row mb-4">
            <div class="col-lg-4 col-md-6 mb-3">
                <div class="card border-0 shadow-sm h-100">
                    <div class="card-body text-center">
                        <div class="d-flex align-items-center justify-content-center mb-3">
                            <div class="bg-warning bg-opacity-10 p-3">
                            <i class="bx bx-time-five rounded fs-2"></i>
                            </div>
                        </div>
                        <h3 class="fw-bold text-warning mb-1" id="pendingCount">{{ $stats['pending'] ?? 0 }}</h3>
                        <p class="text-muted mb-0">{{ __('Pending Requests') }}</p>
                    </div>
                </div>
            </div>
            <div class="col-lg-4 col-md-6 mb-3">
                <div class="card border-0 shadow-sm h-100">
                    <div class="card-body text-center">
                        <div class="d-flex align-items-center justify-content-center mb-3 text-white">
                            <div class="bg-success rounded p-3">
                                <i class='bx bx-check fs-2'></i> 
                            </div>
                        </div>
                        <h3 class="fw-bold text-success mb-1" id="approvedCount">{{ $stats['completed'] ?? 0 }}</h3>
                        <p class="text-muted mb-0">{{ __('Approved') }}</p>
                    </div>
                </div>
            </div>
            <div class="col-lg-4 col-md-6 mb-3">
                <div class="card border-0 shadow-sm h-100">
                    <div class="card-body text-center">
                        <div class="d-flex align-items-center justify-content-center mb-3">
                            <div class="bg-danger rounded p-3 text-white">
                                <i class="bx bx-x-circle fs-2"></i>
                            </div>
                        </div>
                        <h3 class="fw-bold text-danger mb-1" id="rejectedCount">{{ $stats['rejected'] ?? 0 }}</h3>
                        <p class="text-muted mb-0">{{ __('Rejected') }}</p>
                    </div>
                </div>
            </div>
        </div>

        <!-- Requests List -->
        <div class="row">
            <div class="col-12">
                <div class="card border-0 shadow-sm">
                    <div class="card-body p-0">
                        <!-- Loading Indicator -->
                        <div id="loadingIndicator" class="text-center py-5 d-none">
                            <div class="spinner-border text-primary" role="status">
                                <span class="visually-hidden">{{ __('Loading...') }}</span>
                            </div>
                            <p class="mt-2 text-muted">{{ __('Loading requests...') }}</p>
                        </div>

                        <!-- Requests Container -->
                        <div id="requestsContainer">
                            @include('admin.balance-requests.partials.requests-list', ['requests' => $requests])
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Approve Modal -->
<div class="modal fade" id="approveModal" tabindex="-1">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content border-0 shadow">
            <div class="modal-header bg-success text-white">
                <h5 class="modal-title">
                    <i class="bx bx-check-circle me-2"></i>{{ __('Approve Request') }}
                </h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <div class="text-center mb-3">
                    <div class="bg-success bg-opacity-10 text-white rounded-circle d-inline-flex align-items-center justify-content-center mb-3" style="width: 80px; height: 80px;">
                        <i class="bx bx-check-circle fs-"></i>
                    </div>
                </div>
                <p class="text-center">{{ __('Are you sure you want to approve this balance request?') }}</p>
                <div class="alert alert-success">
                    <i class="bx bx-info-circle me-2"></i>
                    <strong>{{ __('User balance will be updated immediately') }}</strong>
                </div>
            </div>
            <div class="modal-footer border-0">
                <button type="button" class="btn btn-light" data-bs-dismiss="modal">{{ __('Cancel') }}</button>
                <button type="button" class="btn btn-success" id="confirmApprove">
                    <i class="bx bx-check me-2"></i>{{ __('Approve') }}
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Reject Modal -->
<div class="modal fade" id="rejectModal" tabindex="-1">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content border-0 shadow">
            <div class="modal-header bg-danger text-white">
                <h5 class="modal-title text-white">
                    <i class="bx bx-x-circle me-2"></i>{{ __('Reject Request') }}
                </h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <div class="text-center mb-3">
                    <div class="bg-danger text-white bg-opacity-10 rounded-circle d-inline-flex align-items-center justify-content-center mb-3" style="width: 80px; height: 80px;">
                        <i class="bx bx-x-circle fs-1"></i>
                    </div>
                </div>
                <div class="mb-3">
                    <label class="form-label fw-bold">{{ __('Rejection Reason') }} <span class="text-danger">*</span></label>
                    <textarea class="form-control" id="rejectionReason" rows="4" required 
                              placeholder="{{ __('Please explain the reason for rejection...') }}"></textarea>
                    <div class="invalid-feedback">
                        {{ __('Please provide a reason for rejection') }}
                    </div>
                </div>
            </div>
            <div class="modal-footer border-0">
                <button type="button" class="btn btn-light" data-bs-dismiss="modal">{{ __('Cancel') }}</button>
                <button type="button" class="btn btn-danger" id="confirmReject">
                    <i class="bx bx-x me-2"></i>{{ __('Reject') }}
                </button>
            </div>
        </div>
    </div>
</div>

<!-- View Request Modal -->
<div class="modal fade" id="viewModal" tabindex="-1">
    <div class="modal-dialog modal-lg modal-dialog-centered">
        <div class="modal-content border-0 shadow">
            <div class="modal-header bg-info text-white">
                <h5 class="modal-title">
                    <i class="bx bx-show me-2"></i>{{ __('Request Details') }}
                </h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body" id="viewModalBody">
                <!-- Content will be loaded here -->
            </div>
            <div class="modal-footer border-0">
                <button type="button" class="btn btn-light" data-bs-dismiss="modal">{{ __('Close') }}</button>
            </div>
        </div>
    </div>
</div>
@endsection

@section('script')
<script>
// Global variables
let currentRequestId = null;

// View request details
function viewRequest(id) {
    $('#viewModalBody').html(`
        <div class="text-center py-4">
            <div class="spinner-border text-primary" role="status">
                <span class="visually-hidden">{{ __("Loading...") }}</span>
            </div>
        </div>
    `);
    
    const modal = new bootstrap.Modal(document.getElementById('viewModal'));
    modal.show();
    
    $.ajax({
        url: `/admin/balance-requests/${id}`,
        type: 'GET',
        headers: {
            'X-Requested-With': 'XMLHttpRequest'
        },
        success: function(response) {
            $('#viewModalBody').html(response.html || response);
        },
        error: function(xhr) {
            let message = '{{ __("Failed to load request details") }}';
            if (xhr.responseJSON && xhr.responseJSON.message) {
                message = xhr.responseJSON.message;
            }
            
            $('#viewModalBody').html(`
                <div class="alert alert-danger">
                    <i class="bx bx-error-circle me-2"></i>${message}
                </div>
            `);
        }
    });
}

// Approve request
function approveRequest(id) {
    currentRequestId = id;
    const modal = new bootstrap.Modal(document.getElementById('approveModal'));
    modal.show();
}

// Reject request
function rejectRequest(id) {
    currentRequestId = id;
    $('#rejectionReason').val('').removeClass('is-invalid');
    const modal = new bootstrap.Modal(document.getElementById('rejectModal'));
    modal.show();
}

// Refresh requests
function refreshRequests() {
    loadRequests(window.location.href);
}

// Filter requests with Ajax
function filterRequests() {
    const status = $('#statusFilter').val();
    const url = new URL(window.location.href);
    
    if (status) {
        url.searchParams.set('status', status);
    } else {
        url.searchParams.delete('status');
    }
    
    loadRequests(url.toString());
}

// Load requests via Ajax
function loadRequests(url) {
    $('#loadingIndicator').removeClass('d-none');
    $('#requestsContainer').addClass('opacity-50');
    
    $.ajax({
        url: url,
        type: 'GET',
        headers: {
            'X-Requested-With': 'XMLHttpRequest'
        },
        success: function(response) {
            if (response.html) {
                $('#requestsContainer').html(response.html);
                
                // Update stats
                if (response.stats) {
                    $('#pendingCount').text(response.stats.pending || 0);
                    $('#approvedCount').text(response.stats.completed || 0);
                    $('#rejectedCount').text(response.stats.rejected || 0);
                }
                
                // Update URL without reload
                window.history.pushState({}, '', url);
                
                Lobibox.notify('success', {
                    pauseDelayOnHover: true,
                    continueDelayOnInactiveTab: false,
                    position: 'top right',
                    icon: 'bx bx-check-circle',
                    msg: '{{ __("Requests updated successfully") }}'
                });
            }
        },
        error: function(xhr) {
            let message = '{{ __("Failed to load requests") }}';
            if (xhr.responseJSON && xhr.responseJSON.message) {
                message = xhr.responseJSON.message;
            }
            
            Lobibox.notify('error', {
                pauseDelayOnHover: true,
                continueDelayOnInactiveTab: false,
                position: 'top right',
                icon: 'bx bx-x-circle',
                msg: message
            });
        },
        complete: function() {
            $('#loadingIndicator').addClass('d-none');
            $('#requestsContainer').removeClass('opacity-50');
        }
    });
}

// Confirm approve
$('#confirmApprove').on('click', function() {
    if (!currentRequestId) return;
    
    const button = $(this);
    const originalText = button.html();
    
    button.prop('disabled', true).html('<i class="bx bx-loader-alt bx-spin me-2"></i>{{ __("Processing...") }}');
    
    $.ajax({
        url: `/admin/balance-requests/${currentRequestId}/approve`,
        type: 'POST',
        data: {
            '_token': $('meta[name="csrf-token"]').attr('content'),
            '_method': 'PATCH'
        },
        headers: {
            'X-Requested-With': 'XMLHttpRequest'
        },
        success: function(response, textStatus, xhr) {
            // Check if it's a redirect (302)
            if (xhr.status === 200 || xhr.status === 302) {
                // Hide modal
                bootstrap.Modal.getInstance(document.getElementById('approveModal')).hide();
                
                // Update the specific request item
                updateRequestItem(currentRequestId, 'completed');
                
                Lobibox.notify('success', {
                    pauseDelayOnHover: true,
                    continueDelayOnInactiveTab: false,
                    position: 'top right',
                    icon: 'bx bx-check-circle',
                    msg: '{{ __("Request approved successfully") }}'
                });
            } else if (response && response.success) {
                // Hide modal
                bootstrap.Modal.getInstance(document.getElementById('approveModal')).hide();
                
                // Update the specific request item
                updateRequestItem(currentRequestId, 'completed');
                
                Lobibox.notify('success', {
                    pauseDelayOnHover: true,
                    continueDelayOnInactiveTab: false,
                    position: 'top right',
                    icon: 'bx bx-check-circle',
                    msg: response.message || '{{ __("Request approved successfully") }}'
                });
            } else {
                throw new Error(response.message || '{{ __("Failed to approve request") }}');
            }
        },
        error: function(xhr) {
            let message = '{{ __("Failed to approve request") }}';
            if (xhr.responseJSON && xhr.responseJSON.message) {
                message = xhr.responseJSON.message;
            }
            
            Lobibox.notify('error', {
                pauseDelayOnHover: true,
                continueDelayOnInactiveTab: false,
                position: 'top right',
                icon: 'bx bx-x-circle',
                msg: message
            });
        },
        complete: function() {
            button.prop('disabled', false).html(originalText);
            currentRequestId = null;
        }
    });
});

// Confirm reject
$('#confirmReject').on('click', function() {
    if (!currentRequestId) return;
    
    const reason = $('#rejectionReason').val().trim();
    if (!reason) {
        $('#rejectionReason').addClass('is-invalid');
        return;
    }
    
    $('#rejectionReason').removeClass('is-invalid');
    
    const button = $(this);
    const originalText = button.html();
    
    button.prop('disabled', true).html('<i class="bx bx-loader-alt bx-spin me-2"></i>{{ __("Processing...") }}');
    
    $.ajax({
        url: `/admin/balance-requests/${currentRequestId}/reject`,
        type: 'POST',
        data: {
            '_token': $('meta[name="csrf-token"]').attr('content'),
            '_method': 'PATCH',
            'rejection_reason': reason
        },
        headers: {
            'X-Requested-With': 'XMLHttpRequest'
        },
        success: function(response, textStatus, xhr) {
            // Check if it's a redirect (302)
            if (xhr.status === 200 || xhr.status === 302) {
                // Hide modal
                bootstrap.Modal.getInstance(document.getElementById('rejectModal')).hide();
                
                // Update the specific request item
                updateRequestItem(currentRequestId, 'rejected');
                
                Lobibox.notify('success', {
                    pauseDelayOnHover: true,
                    continueDelayOnInactiveTab: false,
                    position: 'top right',
                    icon: 'bx bx-check-circle',
                    msg: '{{ __("Request rejected successfully") }}'
                });
            } else if (response && response.success) {
                // Hide modal
                bootstrap.Modal.getInstance(document.getElementById('rejectModal')).hide();
                
                // Update the specific request item
                updateRequestItem(currentRequestId, 'rejected');
                
                Lobibox.notify('success', {
                    pauseDelayOnHover: true,
                    continueDelayOnInactiveTab: false,
                    position: 'top right',
                    icon: 'bx bx-check-circle',
                    msg: response.message || '{{ __("Request rejected successfully") }}'
                });
            } else {
                throw new Error(response.message || '{{ __("Failed to reject request") }}');
            }
        },
        error: function(xhr) {
            let message = '{{ __("Failed to reject request") }}';
            if (xhr.responseJSON && xhr.responseJSON.message) {
                message = xhr.responseJSON.message;
            }
            
            Lobibox.notify('error', {
                pauseDelayOnHover: true,
                continueDelayOnInactiveTab: false,
                position: 'top right',
                icon: 'bx bx-x-circle',
                msg: message
            });
        },
        complete: function() {
            button.prop('disabled', false).html(originalText);
            currentRequestId = null;
        }
    });
});

// Update specific request item status
function updateRequestItem(requestId, newStatus) {
    const requestItem = $(`.request-item[data-id="${requestId}"]`);
    if (requestItem.length) {
        // Get current status for stats update
        const badge = requestItem.find('.badge');
        const currentStatus = badge.text().trim();
        
        // Update badge
        badge.removeClass('bg-warning bg-success bg-danger');
        
        if (newStatus === 'completed') {
            badge.addClass('bg-success').text('{{ __("Approved") }}');
            updateStatsCounters(currentStatus, '{{ __("Approved") }}');
        } else if (newStatus === 'rejected') {
            badge.addClass('bg-danger').text('{{ __("Rejected") }}');
            updateStatsCounters(currentStatus, '{{ __("Rejected") }}');
        }
        
        // Remove action buttons
        requestItem.find('.btn-success, .btn-danger').remove();
        
        // Add fade effect
        requestItem.addClass('table-success').delay(2000).queue(function() {
            $(this).removeClass('table-success').dequeue();
        });
    }
}

// Update stats counters based on status change
function updateStatsCounters(oldStatus, newStatus) {
    let pendingCount = parseInt($('#pendingCount').text()) || 0;
    let approvedCount = parseInt($('#approvedCount').text()) || 0;
    let rejectedCount = parseInt($('#rejectedCount').text()) || 0;
    
    // Decrease from old status
    if (oldStatus === '{{ __("Pending") }}') {
        pendingCount = Math.max(0, pendingCount - 1);
    } else if (oldStatus === '{{ __("Approved") }}') {
        approvedCount = Math.max(0, approvedCount - 1);
    } else if (oldStatus === '{{ __("Rejected") }}') {
        rejectedCount = Math.max(0, rejectedCount - 1);
    }
    
    // Increase to new status
    if (newStatus === '{{ __("Pending") }}') {
        pendingCount++;
    } else if (newStatus === '{{ __("Approved") }}') {
        approvedCount++;
    } else if (newStatus === '{{ __("Rejected") }}') {
        rejectedCount++;
    }
    
    // Update display
    $('#pendingCount').text(pendingCount);
    $('#approvedCount').text(approvedCount);
    $('#rejectedCount').text(rejectedCount);
}

// Update stats counters manually
function updateStats() {
    // Get current counts
    let pendingCount = parseInt($('#pendingCount').text()) || 0;
    let approvedCount = parseInt($('#approvedCount').text()) || 0;
    let rejectedCount = parseInt($('#rejectedCount').text()) || 0;
    
    // Update based on the action performed
    if (currentRequestId) {
        const requestItem = $(`.request-item[data-id="${currentRequestId}"]`);
        const currentStatus = requestItem.find('.badge').text().trim();
        
        // Decrease from current status
        if (currentStatus === '{{ __("Pending") }}') {
            pendingCount = Math.max(0, pendingCount - 1);
        } else if (currentStatus === '{{ __("Approved") }}') {
            approvedCount = Math.max(0, approvedCount - 1);
        } else if (currentStatus === '{{ __("Rejected") }}') {
            rejectedCount = Math.max(0, rejectedCount - 1);
        }
    }
    
    // Update display
    $('#pendingCount').text(pendingCount);
    $('#approvedCount').text(approvedCount);
    $('#rejectedCount').text(rejectedCount);
}

// Add custom styles
const customStyles = `
<style>
.bg-gradient-primary {
    background: linear-gradient(45deg, #007bff, #0056b3) !important;
}
.request-item:hover {
    background-color: rgba(0, 123, 255, 0.05);
}
.table-success {
    background-color: rgba(25, 135, 84, 0.1) !important;
    transition: background-color 0.3s ease;
}
.opacity-50 {
    opacity: 0.5;
    transition: opacity 0.3s ease;
}
@media (max-width: 768px) {
    .btn-group .btn {
        margin-bottom: 5px;
    }
}
</style>
`;
$('head').append(customStyles);

// Initialize page when DOM is ready
$(document).ready(function() {
    // Initialize status filter
    $('#statusFilter').val('{{ request("status") }}');
    
    // Status filter change handler
    $('#statusFilter').on('change', function() {
        filterRequests();
    });
});
</script>
@endsection