@extends('layouts.app')

@section('title', __('Language Management'))

@section('wrapper')
<div class="page-wrapper">
    <div class="page-content">
        <!-- Breadcrumb -->
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-3">
            <div class="breadcrumb-title pe-3">{{__('Management')}}</div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ url('/home') }}"><i class="bx bx-home-alt"></i></a></li>
                        <li class="breadcrumb-item active" aria-current="page">{{__('Language Management')}}</li>
                    </ol>
                </nav>
            </div>
        </div>

        <!-- Stats Cards -->
        <div class="row mb-4">
            <div class="col-12 col-md-4">
                <div class="card radius-10 border-start border-0 border-4 border-info">
                    <div class="card-body">
                        <div class="d-flex align-items-center">
                            <div>
                                <p class="mb-0 text-secondary">{{__('Total Languages')}}</p>
                                <h4 class="my-1 text-info">{{ $stats['total_languages'] }}</h4>
                            </div>
                            <div class="widgets-icons-2 rounded-circle bg-gradient-blues text-white ms-auto">
                                <i class='bx bx-globe'></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-12 col-md-4">
                <div class="card radius-10 border-start border-0 border-4 border-success">
                    <div class="card-body">
                        <div class="d-flex align-items-center">
                            <div>
                                <p class="mb-0 text-secondary">{{__('Translation Keys')}}</p>
                                <h4 class="my-1 text-success">{{ $stats['total_keys'] }}</h4>
                            </div>
                            <div class="widgets-icons-2 rounded-circle bg-gradient-greens text-white ms-auto">
                                <i class='bx bx-key'></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-12 col-md-4">
                <div class="card radius-10 border-start border-0 border-4 border-warning">
                    <div class="card-body">
                        <div class="d-flex align-items-center">
                            <div>
                                <p class="mb-0 text-secondary">{{__('Avg. Completion')}}</p>
                                <h4 class="my-1 text-warning">{{ $stats['completion_rate'] }}%</h4>
                            </div>
                            <div class="widgets-icons-2 rounded-circle bg-gradient-orange text-white ms-auto">
                                <i class='bx bx-chart'></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Main Content -->
        <div class="card">
            <div class="card-header">
                <div class="d-flex align-items-center justify-content-between">
                    <h5 class="mb-0">{{__('Language Management')}}</h5>
                    <button type="button" class="btn btn-primary btn-sm" data-bs-toggle="offcanvas" data-bs-target="#addLanguageOffcanvas">
                        <i class="bx bx-plus"></i> {{__('Add Language')}}
                    </button>
                </div>
            </div>
            <div class="card-body">
                <!-- Loading Spinner -->
                <div id="loading" class="text-center py-4" style="display: none;">
                    <div class="spinner-border text-primary" role="status">
                        <span class="visually-hidden">{{__('Loading...')}}</span>
                    </div>
                </div>

                <!-- Languages Grid -->
                <div id="languagesGrid" class="row">
                    <!-- Languages will be loaded here via AJAX -->
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Add Language Offcanvas -->
<div class="offcanvas offcanvas-end" tabindex="-1" id="addLanguageOffcanvas" aria-labelledby="addLanguageOffcanvasLabel">
    <div class="offcanvas-header">
        <h5 class="offcanvas-title" id="addLanguageOffcanvasLabel">{{__('Add New Language')}}</h5>
        <button type="button" class="btn-close" data-bs-dismiss="offcanvas" aria-label="Close"></button>
    </div>
    <div class="offcanvas-body">
        <form id="addLanguageForm">
            <div class="mb-3">
                <label for="code" class="form-label">{{__('Language Code')}} *</label>
                <input type="text" class="form-control" id="code" name="code" placeholder="en, tr, ar..." maxlength="10" required>
                <div class="form-text">{{__('2-3 character language code')}}</div>
            </div>
            <div class="mb-3">
                <label for="order" class="form-label">{{__('Display Order')}}</label>
                <input type="number" class="form-control" id="order" name="order" min="1" value="999">
            </div>
            <div class="mb-3">
                <label for="name" class="form-label">{{__('Language Name')}} *</label>
                <input type="text" class="form-control" id="name" name="name" placeholder="English" required>
            </div>
            <div class="mb-3">
                <label for="native_name" class="form-label">{{__('Native Name')}} *</label>
                <input type="text" class="form-control" id="native_name" name="native_name" placeholder="English" required>
            </div>
            <div class="mb-3">
                <label for="direction" class="form-label">{{__('Text Direction')}} *</label>
                <select class="form-select" id="direction" name="direction" required>
                    <option value="ltr">{{__('Left to Right (LTR)')}}</option>
                    <option value="rtl">{{__('Right to Left (RTL)')}}</option>
                </select>
            </div>
            <div class="mb-3">
                <label for="icon" class="form-label">{{__('Icon/Flag')}}</label>
                <input type="text" class="form-control" id="icon" name="icon" placeholder="🇺🇸, en, flag-icon...">
            </div>
            <div class="d-grid gap-2">
                <button type="submit" class="btn btn-primary">
                    <span class="spinner-border spinner-border-sm me-2" style="display: none;"></span>
                    {{__('Create Language')}}
                </button>
                <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="offcanvas">{{__('Cancel')}}</button>
            </div>
        </form>
    </div>
</div>

<!-- Edit Language Offcanvas -->
<div class="offcanvas offcanvas-end" tabindex="-1" id="editLanguageOffcanvas" aria-labelledby="editLanguageOffcanvasLabel">
    <div class="offcanvas-header">
        <h5 class="offcanvas-title" id="editLanguageOffcanvasLabel">{{__('Edit Language')}}</h5>
        <button type="button" class="btn-close" data-bs-dismiss="offcanvas" aria-label="Close"></button>
    </div>
    <div class="offcanvas-body">
        <form id="editLanguageForm">
            <input type="hidden" id="edit_code" name="code">
            <div class="mb-3">
                <label class="form-label">{{__('Language Code')}}</label>
                <input type="text" class="form-control" id="edit_code_display" readonly>
            </div>
            <div class="mb-3">
                <label for="edit_order" class="form-label">{{__('Display Order')}}</label>
                <input type="number" class="form-control" id="edit_order" name="order" min="1">
            </div>
            <div class="mb-3">
                <label for="edit_name" class="form-label">{{__('Language Name')}} *</label>
                <input type="text" class="form-control" id="edit_name" name="name" required>
            </div>
            <div class="mb-3">
                <label for="edit_native_name" class="form-label">{{__('Native Name')}} *</label>
                <input type="text" class="form-control" id="edit_native_name" name="native_name" required>
            </div>
            <div class="mb-3">
                <label for="edit_direction" class="form-label">{{__('Text Direction')}} *</label>
                <select class="form-select" id="edit_direction" name="direction" required>
                    <option value="ltr">{{__('Left to Right (LTR)')}}</option>
                    <option value="rtl">{{__('Right to Left (RTL)')}}</option>
                </select>
            </div>
            <div class="mb-3">
                <label for="edit_icon" class="form-label">{{__('Icon/Flag')}}</label>
                <input type="text" class="form-control" id="edit_icon" name="icon">
            </div>
            <div class="d-grid gap-2">
                <button type="submit" class="btn btn-primary">
                    <span class="spinner-border spinner-border-sm me-2" style="display: none;"></span>
                    {{__('Update Language')}}
                </button>
                <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="offcanvas">{{__('Cancel')}}</button>
            </div>
        </form>
    </div>
</div>

<!-- Delete Language Confirmation Offcanvas -->
<div class="offcanvas offcanvas-end" tabindex="-1" id="deleteLanguageOffcanvas" aria-labelledby="deleteLanguageOffcanvasLabel">
    <div class="offcanvas-header border-bottom">
        <h5 class="offcanvas-title text-danger" id="deleteLanguageOffcanvasLabel">
            <i class="bx bx-trash me-2"></i>{{__('Delete Language')}}
        </h5>
        <button type="button" class="btn-close" data-bs-dismiss="offcanvas" aria-label="Close"></button>
    </div>
    <div class="offcanvas-body">
        <div class="text-center mb-4">
            <div class="avatar avatar-xl bg-danger-subtle text-danger rounded-circle mx-auto mb-3">
                <i class="bx bx-trash fs-1"></i>
            </div>
            <h6 class="mb-2">{{__('Are you sure?')}}</h6>
            <p class="text-muted mb-0">{{__('This action cannot be undone. All translations for this language will be permanently deleted.')}}</p>
        </div>
        
        <div class="card border-danger">
            <div class="card-body">
                <div class="d-flex align-items-center">
                    <div class="language-icon me-3">
                        <span class="fs-4" id="delete-language-icon">🌐</span>
                    </div>
                    <div>
                        <h6 class="mb-1" id="delete-language-name">Language Name</h6>
                        <small class="text-muted">
                            <span id="delete-language-code">CODE</span> • 
                            <span id="delete-language-keys">0</span> {{__('translations')}}
                        </small>
                    </div>
                </div>
            </div>
        </div>
        
        <div class="alert alert-danger mt-3" role="alert">
            <div class="d-flex align-items-start">
                <i class="bx bx-error-circle fs-5 me-2 mt-1"></i>
                <div>
                    <h6 class="alert-heading mb-1">{{__('Warning!')}}</h6>
                    <p class="mb-0 small">{{__('This will permanently delete the language file and all its translations. This action cannot be undone.')}}</p>
                </div>
            </div>
        </div>
        
        <div class="mb-3">
            <label for="delete-confirmation-text" class="form-label">
                {{__('Type')}} <code>DELETE</code> {{__('to confirm')}}:
            </label>
            <input type="text" class="form-control" id="delete-confirmation-text" placeholder="DELETE">
        </div>
        
        <div class="d-grid gap-2">
            <button type="button" class="btn btn-danger" id="confirm-delete-btn" disabled onclick="confirmDeleteLanguage()">
                <span class="spinner-border spinner-border-sm me-2" style="display: none;"></span>
                <i class="bx bx-trash me-2"></i>
                {{__('Yes, Delete Language')}}
            </button>
            <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="offcanvas">
                <i class="bx bx-x me-2"></i>
                {{__('Cancel')}}
            </button>
        </div>
    </div>
</div>
@endsection

@section('script')
<script>
$(document).ready(function() {
    // CSRF Token
    $.ajaxSetup({
        headers: {
            'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
        }
    });

    // Load languages on page load
    loadLanguages();

    // Add Language Form Submit
    $('#addLanguageForm').on('submit', function(e) {
        e.preventDefault();
        e.stopPropagation();
        
        const form = $(this);
        const submitBtn = form.find('button[type="submit"]');
        const spinner = submitBtn.find('.spinner-border');
        
        // Show loading
        submitBtn.prop('disabled', true);
        spinner.show();
        
        $.ajax({
            url: '{{ route("admin.languages.store") }}',
            method: 'POST',
            data: form.serialize(),
            success: function(response) {
                if (response.success) {
                    bootstrap.Offcanvas.getInstance('#addLanguageOffcanvas').hide();
                    form[0].reset();
                    loadLanguages();
                    showToast('success', response.message);
                }
            },
            error: function(xhr) {
                const errors = xhr.responseJSON?.errors || {};
                showValidationErrors(form, errors);
                if (xhr.responseJSON?.message) {
                    showToast('error', xhr.responseJSON.message);
                }
            },
            complete: function() {
                submitBtn.prop('disabled', false);
                spinner.hide();
            }
        });
        
        return false;
    });

    // Edit Language Form Submit
    $('#editLanguageForm').on('submit', function(e) {
        e.preventDefault();
        e.stopPropagation();
        
        const form = $(this);
        const submitBtn = form.find('button[type="submit"]');
        const spinner = submitBtn.find('.spinner-border');
        const code = $('#edit_code').val();
        
        // Show loading
        submitBtn.prop('disabled', true);
        spinner.show();
        
        $.ajax({
            url: '/admin/languages/' + code,
            method: 'POST',
            data: form.serialize() + '&_method=PUT&_token=' + $('meta[name="csrf-token"]').attr('content'),
            success: function(response) {
                if (response.success) {
                    bootstrap.Offcanvas.getInstance('#editLanguageOffcanvas').hide();
                    loadLanguages();
                    showToast('success', response.message);
                }
            },
            error: function(xhr) {
                const errors = xhr.responseJSON?.errors || {};
                showValidationErrors(form, errors);
                if (xhr.responseJSON?.message) {
                    showToast('error', xhr.responseJSON.message);
                }
            },
            complete: function() {
                submitBtn.prop('disabled', false);
                spinner.hide();
            }
        });
        
        return false;
    });

    // Load Languages Function
    function loadLanguages() {
        $('#loading').show();
        $('#languagesGrid').hide();
        
        $.ajax({
            url: '{{ route("admin.languages.data") }}',
            method: 'GET',
            success: function(response) {
                if (response.success) {
                    renderLanguages(response.data);
                }
            },
            error: function() {
                showToast('error', '{{__("An error occurred while loading languages")}}');
            },
            complete: function() {
                $('#loading').hide();
                $('#languagesGrid').show();
            }
        });
    }

    // Render Languages Function
    function renderLanguages(languages) {
        let html = '';
        
        languages.forEach(function(lang) {
            const completion = calculateCompletion(lang);
            const progressColor = completion >= 80 ? 'success' : completion >= 50 ? 'warning' : 'danger';
            
            html += `
                <div class="col-12 col-md-6 col-lg-4 mb-3">
                    <div class="card h-100">
                        <div class="card-body">
                            <div class="d-flex align-items-center mb-3">
                                <div class="language-icon me-3">
                                    <span class="fs-4">${lang.icon || '🌐'}</span>
                                </div>
                                <div class="flex-grow-1">
                                    <h6 class="mb-0">${lang.native_name}</h6>
                                    <small class="text-muted">${lang.code.toUpperCase()} • ${lang.direction.toUpperCase()}</small>
                                </div>
                                <div class="dropdown">
                                    <button class="btn btn-sm btn-outline-secondary dropdown-toggle" type="button" data-bs-toggle="dropdown">
                                        <i class="bx bx-dots-vertical-rounded"></i>
                                    </button>
                                    <ul class="dropdown-menu">
                                        <li><a class="dropdown-item" href="javascript:void(0)" onclick="editLanguage('${lang.code}'); return false;">
                                            <i class="bx bx-edit me-2"></i>{{__('Edit')}}
                                        </a></li>
                                        <li><a class="dropdown-item" href="/admin/languages/${lang.code}/translations">
                                      <i class='bx  bx-folder-down-arrow'></i>{{__('Translations')}}
                                        </a></li>
                                        <li><a class="dropdown-item" href="#" onclick="exportLanguage('${lang.code}')">
                                            <i class="bx bx-download me-2"></i>{{__('Export')}}
                                        </a></li>
                                        ${lang.code !== 'en' ? `
                                        <li><hr class="dropdown-divider"></li>
                                        <li><a class="dropdown-item text-danger" href="javascript:void(0)" onclick="deleteLanguage('${lang.code}')">
                                            <i class="bx bx-trash me-2"></i>{{__('Delete')}}
                                        </a></li>
                                        ` : ''}
                                    </ul>
                                </div>
                            </div>
                            
                            <div class="mb-2">
                                <div class="d-flex justify-content-between align-items-center mb-1">
                                    <small class="text-muted">{{__('Completion')}}</small>
                                    <small class="fw-bold">${completion}%</small>
                                </div>
                                <div class="progress" style="height: 6px;">
                                    <div class="progress-bar bg-${progressColor}" style="width: ${completion}%"></div>
                                </div>
                            </div>
                            
                            <div class="d-flex justify-content-between text-muted small">
                                <span>{{__('Keys')}}: ${lang.key_count}</span>
                                <span>{{__('Order')}}: ${lang.order}</span>
                            </div>
                        </div>
                    </div>
                </div>
            `;
        });
        
        $('#languagesGrid').html(html);
    }

    // Calculate completion percentage
    function calculateCompletion(lang) {
        return lang.completion || 0;
    }

    // Edit Language Function
    window.editLanguage = function(code) {
        $.ajax({
            url: '{{ route("admin.languages.data") }}',
            method: 'GET',
            success: function(response) {
                if (response.success) {
                    const lang = response.data.find(l => l.code === code);
                    if (lang) {
                        $('#edit_code').val(lang.code);
                        $('#edit_code_display').val(lang.code.toUpperCase());
                        $('#edit_name').val(lang.name);
                        $('#edit_native_name').val(lang.native_name);
                        $('#edit_direction').val(lang.direction);
                        $('#edit_icon').val(lang.icon);
                        $('#edit_order').val(lang.order);
                        new bootstrap.Offcanvas('#editLanguageOffcanvas').show();
                    }
                }
            }
        });
    };

    // Delete Language Function
    window.deleteLanguage = function(code) {
        // Get language data
        $.ajax({
            url: '{{ route("admin.languages.data") }}',
            method: 'GET',
            success: function(response) {
                if (response.success) {
                    const lang = response.data.find(l => l.code === code);
                    if (lang) {
                        // Populate delete confirmation offcanvas
                        $('#delete-language-icon').text(lang.icon || '🌐');
                        $('#delete-language-name').text(lang.native_name);
                        $('#delete-language-code').text(lang.code.toUpperCase());
                        $('#delete-language-keys').text(lang.key_count);
                        
                        // Store language code for deletion
                        $('#deleteLanguageOffcanvas').data('language-code', code);
                        
                        // Reset confirmation input and button
                        $('#delete-confirmation-text').val('');
                        $('#confirm-delete-btn').prop('disabled', true);
                        
                        // Show offcanvas
                        new bootstrap.Offcanvas('#deleteLanguageOffcanvas').show();
                    }
                }
            }
        });
    };

    // Confirm Delete Language Function
    window.confirmDeleteLanguage = function() {
        const code = $('#deleteLanguageOffcanvas').data('language-code');
        const confirmBtn = $('#confirm-delete-btn');
        const spinner = confirmBtn.find('.spinner-border');
        
        // Show loading
        confirmBtn.prop('disabled', true);
        spinner.show();
        
        $.ajax({
            url: '/admin/languages/' + code,
            method: 'POST',
            data: '_method=DELETE&_token=' + $('meta[name="csrf-token"]').attr('content'),
            success: function(response) {
                if (response.success) {
                    bootstrap.Offcanvas.getInstance('#deleteLanguageOffcanvas').hide();
                    loadLanguages();
                    showToast('success', response.message);
                }
            },
            error: function(xhr) {
                showToast('error', xhr.responseJSON?.message || '{{__("An error occurred")}}');
            },
            complete: function() {
                confirmBtn.prop('disabled', false);
                spinner.hide();
            }
        });
    };

    // Export Language Function
    window.exportLanguage = function(code) {
        window.location.href = '/admin/languages/' + code + '/export';
    };

    // Show Toast Notification
    function showToast(type, message) {
        if (typeof Lobibox !== 'undefined') {
            if (type === 'success') {
                Lobibox.notify('success', {
                    pauseDelayOnHover: true,
                    continueDelayOnInactiveTab: false,
                    position: 'top right',
                    icon: 'bx bx-check-circle',
                    msg: message
                });
            } else if (type === 'error') {
                Lobibox.notify('error', {
                    pauseDelayOnHover: true,
                    continueDelayOnInactiveTab: false,
                    position: 'top right',
                    icon: 'bx bx-x-circle',
                    msg: message
                });
            }
        } else {
            // Fallback to alert if Lobibox is not available
            if (type === 'success') {
                alert('✅ ' + message);
            } else {
                alert('❌ ' + message);
            }
        }
    }

    // Show Validation Errors
    function showValidationErrors(form, errors) {
        // Clear previous errors
        form.find('.is-invalid').removeClass('is-invalid');
        form.find('.invalid-feedback').remove();
        
        // Show new errors
        Object.keys(errors).forEach(function(field) {
            const input = form.find(`[name="${field}"]`);
            input.addClass('is-invalid');
            input.after(`<div class="invalid-feedback">${errors[field][0]}</div>`);
        });
    }

    // Delete confirmation input validation
    $('#delete-confirmation-text').on('input', function() {
        const confirmBtn = $('#confirm-delete-btn');
        const inputValue = $(this).val().trim();
        
        if (inputValue === 'DELETE') {
            confirmBtn.prop('disabled', false).removeClass('btn-outline-danger').addClass('btn-danger');
        } else {
            confirmBtn.prop('disabled', true).removeClass('btn-danger').addClass('btn-outline-danger');
        }
    });

    // Clear validation errors when offcanvas is hidden
    $('.offcanvas').on('hidden.bs.offcanvas', function() {
        $(this).find('.is-invalid').removeClass('is-invalid');
        $(this).find('.invalid-feedback').remove();
        
        // Reset forms if they exist
        if ($(this).find('form').length > 0) {
            $(this).find('form')[0].reset();
        }
        
        $(this).find('.spinner-border').hide();
        $(this).find('button[type="submit"]').prop('disabled', false);
        
        // Reset delete confirmation
        if ($(this).attr('id') === 'deleteLanguageOffcanvas') {
            $('#delete-confirmation-text').val('');
            $('#confirm-delete-btn').prop('disabled', true).removeClass('btn-danger').addClass('btn-outline-danger');
        }
    });

    // Prevent offcanvas backdrop click from closing during form submission
    $('.offcanvas').on('hide.bs.offcanvas', function(e) {
        if ($(this).find('button[type="submit"]').prop('disabled')) {
            e.preventDefault();
            return false;
        }
    });
});
</script>
@endsection