@extends('layouts.app')

@section('style')
    <link href="{{ asset('assets/plugins/notifications/css/lobibox.min.css') }}" rel="stylesheet" />
    <style>
        .account-card {
            border: 1px solid #e9ecef;
            border-radius: 12px;
            transition: all 0.3s ease;
            background: white;
            box-shadow: 0 2px 4px rgba(0,0,0,0.05);
        }
        .account-card:hover {
            box-shadow: 0 8px 25px rgba(0,0,0,0.15);
            transform: translateY(-2px);
        }
        .platform-icon {
            width: 50px;
            height: 50px;
            border-radius: 12px;
            object-fit: cover;
            border: 2px solid #f8f9fa;
        }
        .user-avatar {
            width: 35px;
            height: 35px;
            border-radius: 50%;
            background: linear-gradient(45deg, #667eea, #764ba2);
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: bold;
            font-size: 0.8rem;
        }
        .status-badge {
            font-size: 0.75rem;
            padding: 6px 12px;
            border-radius: 20px;
            font-weight: 600;
        }
        .action-buttons {
            display: flex;
            gap: 8px;
            flex-wrap: wrap;
        }
        .action-buttons .btn {
            padding: 6px 12px;
            font-size: 0.8rem;
            border-radius: 8px;
        }
        .search-container {
            background: linear-gradient(135deg, #dc3545 0%, #fd7e14 100%);
            border-radius: 12px;
            padding: 25px;
            margin-bottom: 25px;
            color: white;
        }
        .search-input {
            border: 2px solid rgba(255,255,255,0.3);
            border-radius: 12px;
            padding: 12px 16px;
            background: rgba(255,255,255,0.9);
            backdrop-filter: blur(10px);
        }
        .search-input:focus {
            border-color: rgba(255,255,255,0.8);
            background: white;
            box-shadow: 0 4px 15px rgba(0,0,0,0.1);
        }
        .filter-btn {
            border-radius: 12px;
            padding: 12px 20px;
            border: 2px solid rgba(255,255,255,0.3);
            background: rgba(255,255,255,0.1);
            color: white;
            backdrop-filter: blur(10px);
            font-weight: 600;
        }
        .filter-btn:hover {
            background: rgba(255,255,255,0.2);
            border-color: rgba(255,255,255,0.5);
            color: white;
            transform: translateY(-1px);
        }
        .loading-spinner {
            display: none;
            text-align: center;
            padding: 40px;
        }
        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: #6c757d;
        }
        .empty-state i {
            font-size: 4rem;
            margin-bottom: 20px;
            opacity: 0.5;
        }
        @media (max-width: 768px) {
            .account-card {
                margin-bottom: 15px;
            }
            .action-buttons {
                justify-content: center;
                margin-top: 15px;
            }
            .search-container {
                padding: 20px;
                margin-bottom: 20px;
            }
            .platform-icon {
                width: 40px;
                height: 40px;
            }
        }
        .pending-highlight {
            border-left: 4px solid #ffc107;
        }
        .approved-highlight {
            border-left: 4px solid #28a745;
        }
        .rejected-highlight {
            border-left: 4px solid #dc3545;
        }
    </style>
@endsection

@section('wrapper')
<div class="page-wrapper">
    <div class="page-content">
        <!-- Breadcrumb -->
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-3">
            <div class="breadcrumb-title pe-3">{{ __('Admin Panel') }}</div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ url('/home') }}"><i class="bx bx-home-alt"></i></a></li>
                        <li class="breadcrumb-item active" aria-current="page">{{ __('Social Media Accounts') }}</li>
                    </ol>
                </nav>
            </div>
        </div>

        <!-- Search and Filter Container -->
        <div class="search-container">
            <div class="row align-items-center">
                <div class="col-md-8 mb-3 mb-md-0">
                    <h4 class="mb-2 text-white">{{ __('Social Media Accounts Management') }}</h4>
                    <p class="mb-0 opacity-75">{{ __('Review and manage user social media accounts') }}</p>
                </div>
                <div class="col-md-4">
                    <div class="text-md-end">
                        <div class="d-flex gap-2 justify-content-md-end">
                            <span class="badge bg-warning fs-6 px-3 py-2" id="pendingCount">0 {{ __('Pending') }}</span>
                            <span class="badge bg-success fs-6 px-3 py-2" id="approvedCount">0 {{ __('Approved') }}</span>
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="row mt-4 g-3">
                <div class="col-12">
                    <div class="input-group">
                        <span class="input-group-text bg-white border-end-0">
                            <i class="bx bx-search text-muted"></i>
                        </span>
                        <input type="text" id="searchInput" class="form-control search-input border-start-0" 
                               placeholder="{{ __('Search by user, platform or username...') }}">
                        <button class="btn btn-primary filter-btn" type="button" onclick="searchAccounts()">
                            {{ __('Search') }}
                        </button>
                    </div>
                </div>
                <div class="col-6 col-md-6">
                    <select id="statusFilter" class="form-select search-input" onchange="searchAccounts()">
                        <option value="">{{ __('All Status') }}</option>
                        <option value="0">{{ __('Pending') }}</option>
                        <option value="1">{{ __('Approved') }}</option>
                        <option value="2">{{ __('Rejected') }}</option>
                    </select>
                </div>
                <div class="col-6 col-md-6">
                    <select id="perPageSelect" class="form-select search-input" onchange="changePerPage()">
                        <option value="12">12 {{ __('per page') }}</option>
                        <option value="24">24 {{ __('per page') }}</option>
                        <option value="48">48 {{ __('per page') }}</option>
                    </select>
                </div>
            </div>
        </div>

        <!-- Loading Spinner -->
        <div class="loading-spinner" id="loadingSpinner">
            <div class="spinner-border text-primary" role="status" style="width: 3rem; height: 3rem;">
                <span class="visually-hidden">{{ __('Loading...') }}</span>
            </div>
            <p class="mt-3">{{ __('Loading accounts...') }}</p>
        </div>

        <!-- Accounts Container -->
        <div id="accountsContainer">
            <!-- Accounts will be loaded here via AJAX -->
        </div>

        <!-- Pagination -->
        <div id="paginationContainer" class="d-flex justify-content-center mt-4">
            <!-- Pagination will be loaded here -->
        </div>

    </div>
</div>
@endsection

@section('script')
<script src="{{ asset('assets/plugins/notifications/js/lobibox.min.js') }}"></script>
<script src="{{ asset('assets/plugins/notifications/js/notifications.min.js') }}"></script>
<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>

<script>
    let currentPage = 1;
    let currentSearch = '';
    let currentStatus = '';
    let currentPerPage = 12;

    // Page load
    $(document).ready(function() {
        loadAccounts();
        
        // Enter key search
        $('#searchInput').on('keypress', function(e) {
            if (e.which == 13) {
                searchAccounts();
            }
        });

        // Real-time search with debounce
        let searchTimeout;
        $('#searchInput').on('input', function() {
            clearTimeout(searchTimeout);
            searchTimeout = setTimeout(function() {
                searchAccounts();
            }, 500);
        });
    });

    function loadAccounts(page = 1) {
        $('#loadingSpinner').show();
        $('#accountsContainer').hide();
        
        $.ajax({
            url: '{{ route("admin.social_accounts.search") }}',
            type: 'GET',
            data: {
                page: page,
                search: currentSearch,
                status: currentStatus,
                per_page: currentPerPage
            },
            success: function(response) {
                if (response.success) {
                    displayAccounts(response.data);
                    displayPagination(response.pagination);
                    updateCounts(response.data);
                    currentPage = page;
                } else {
                    showNotification('error', '{{ __("No data received from server.") }}');
                }
                $('#loadingSpinner').hide();
                $('#accountsContainer').show();
            },
            error: function(xhr, status, error) {
                $('#loadingSpinner').hide();
                showNotification('error', '{{ __("An error occurred while loading accounts.") }}');
            }
        });
    }

    function updateCounts(accounts) {
        const pendingCount = accounts.filter(acc => acc.status === 0).length;
        const approvedCount = accounts.filter(acc => acc.status === 1).length;
        
        $('#pendingCount').text(`${pendingCount} {{ __('Pending') }}`);
        $('#approvedCount').text(`${approvedCount} {{ __('Approved') }}`);
    }

    function displayAccounts(accounts) {
        let html = '';
        
        if (accounts.length === 0) {
            html = `
                <div class="empty-state">
                    <i class="bx bx-user-check"></i>
                    <h4>{{ __('No accounts found') }}</h4>
                    <p>{{ __('No social media accounts found matching your criteria.') }}</p>
                </div>
            `;
        } else {
            html = '<div class="row">';
            accounts.forEach(function(account) {
                html += generateAccountCard(account);
            });
            html += '</div>';
        }
        
        $('#accountsContainer').html(html);
    }

    function generateAccountCard(account) {
        const statusInfo = getStatusInfo(account.status);
        const createdDate = new Date(account.created_at).toLocaleDateString();
        
        // Güvenli user bilgisi alma
        const userName = account.user && account.user.account_holder ? account.user.account_holder : 'Unknown User';
        const userUsername = account.user && account.user.username ? account.user.username : 'unknown';
        const userInitial = userName.charAt(0).toUpperCase();
        
        // Güvenli platform bilgisi alma
        const platformName = account.platform && account.platform.platform ? account.platform.platform : 'Unknown Platform';
        const platformIcon = account.platform && account.platform.icon ? account.platform.icon : 'placeholder.png';
        
        let highlightClass = '';
        switch(account.status) {
            case 0: highlightClass = 'pending-highlight'; break;
            case 1: highlightClass = 'approved-highlight'; break;
            case 2: highlightClass = 'rejected-highlight'; break;
        }

        return `
            <div class="col-lg-4 col-md-6 mb-4">
                <div class="account-card ${highlightClass} p-4 h-100">
                    <div class="d-flex align-items-center mb-3">
                        <img src="/assets/images/app/${platformIcon}" 
                             class="platform-icon me-3" 
                             alt="${platformName}"
                             onerror="this.src='/assets/images/placeholder.png'">
                        <div class="flex-grow-1">
                            <h5 class="mb-1">${platformName}</h5>
                            <small class="text-muted">@${account.username || 'N/A'}</small>
                        </div>
                        <span class="badge ${statusInfo.class} status-badge">${statusInfo.text}</span>
                    </div>
                    
                    <div class="d-flex align-items-center mb-3">
                        <div class="user-avatar me-2">${userInitial}</div>
                        <div>
                            <div class="fw-bold">${userName}</div>
                            <small class="text-muted">#${userUsername}</small>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <small class="text-muted">
                            <i class="bx bx-calendar me-1"></i>
                            {{ __('Submitted') }}: ${createdDate}
                        </small>
                    </div>
                    
                    <div class="action-buttons mt-auto">
                        ${account.status === 0 ? `
                            <button class="btn btn-success btn-sm" onclick="approveAccount(${account.id}, '${platformName.replace(/'/g, "\\'")}', '@${account.username}')">
                                <i class="bx bx-check"></i> {{ __('Approve') }}
                            </button>
                            <button class="btn btn-danger btn-sm" onclick="rejectAccount(${account.id}, '${platformName.replace(/'/g, "\\'")}', '@${account.username}')">
                                <i class="bx bx-x"></i> {{ __('Reject') }}
                            </button>
                        ` : ''}
                        <button class="btn btn-outline-danger btn-sm" onclick="deleteAccount(${account.id}, '${platformName.replace(/'/g, "\\'")}', '@${account.username}')">
                            <i class="bx bx-trash"></i> {{ __('Delete') }}
                        </button>
                    </div>
                </div>
            </div>
        `;
    }

    function getStatusInfo(status) {
        switch(status) {
            case 0:
                return { class: 'bg-warning', text: '{{ __("Pending") }}' };
            case 1:
                return { class: 'bg-success', text: '{{ __("Approved") }}' };
            case 2:
                return { class: 'bg-danger', text: '{{ __("Rejected") }}' };
            default:
                return { class: 'bg-secondary', text: '{{ __("Unknown") }}' };
        }
    }

    function displayPagination(pagination) {
        if (pagination.last_page <= 1) {
            $('#paginationContainer').html('');
            return;
        }

        let html = '<nav><ul class="pagination">';
        
        if (pagination.current_page > 1) {
            html += `<li class="page-item"><a class="page-link" href="#" onclick="loadAccounts(${pagination.current_page - 1})">{{ __('Previous') }}</a></li>`;
        }
        
        for (let i = Math.max(1, pagination.current_page - 2); i <= Math.min(pagination.last_page, pagination.current_page + 2); i++) {
            html += `<li class="page-item ${i === pagination.current_page ? 'active' : ''}">
                        <a class="page-link" href="#" onclick="loadAccounts(${i})">${i}</a>
                     </li>`;
        }
        
        if (pagination.current_page < pagination.last_page) {
            html += `<li class="page-item"><a class="page-link" href="#" onclick="loadAccounts(${pagination.current_page + 1})">{{ __('Next') }}</a></li>`;
        }
        
        html += '</ul></nav>';
        
        html += `<div class="text-center mt-2">
                    <small class="text-muted">
                        {{ __('Showing') }} ${pagination.from} {{ __('to') }} ${pagination.to} {{ __('of') }} ${pagination.total} {{ __('results') }}
                    </small>
                 </div>`;
        
        $('#paginationContainer').html(html);
    }

    function searchAccounts() {
        currentSearch = $('#searchInput').val().trim();
        currentStatus = $('#statusFilter').val();
        currentPage = 1;
        loadAccounts(1);
    }

    function changePerPage() {
        currentPerPage = $('#perPageSelect').val();
        currentPage = 1;
        loadAccounts(1);
    }

    function approveAccount(accountId, platformName, username) {
        Swal.fire({
            title: '{{ __("Approve Account") }}',
            html: `
                <div class="text-center">
                    <i class="bx bx-check-circle text-success" style="font-size: 3rem;"></i>
                    <h5 class="mt-3">{{ __("Approve Social Media Account") }}</h5>
                    <p class="text-muted">{{ __("Do you want to approve") }} <strong>${platformName}</strong> {{ __("account") }} <strong>${username}</strong>?</p>
                </div>
            `,
            icon: false,
            showCancelButton: true,
            confirmButtonColor: '#28a745',
            cancelButtonColor: '#6c757d',
            confirmButtonText: '<i class="bx bx-check me-1"></i>{{ __("Yes, approve it!") }}',
            cancelButtonText: '{{ __("Cancel") }}'
        }).then((result) => {
            if (result.isConfirmed) {
                $.ajax({
                    url: `{{ url('admin/social-accounts') }}/${accountId}/approve`,
                    type: 'POST',
                    data: {
                        _token: '{{ csrf_token() }}'
                    },
                    success: function(response) {
                        if (response.success) {
                            Swal.fire({
                                title: '{{ __("Approved!") }}',
                                text: response.message,
                                icon: 'success',
                                timer: 2000,
                                showConfirmButton: false
                            });
                            loadAccounts(currentPage);
                        } else {
                            showNotification('error', response.message);
                        }
                    },
                    error: function(xhr) {
                        const response = xhr.responseJSON;
                        showNotification('error', response?.message || '{{ __("An error occurred.") }}');
                    }
                });
            }
        });
    }

    function rejectAccount(accountId, platformName, username) {
        Swal.fire({
            title: '{{ __("Reject Account") }}',
            html: `
                <div class="text-center">
                    <i class="bx bx-x-circle text-danger" style="font-size: 3rem;"></i>
                    <h5 class="mt-3">{{ __("Reject Social Media Account") }}</h5>
                    <p class="text-muted">{{ __("Do you want to reject") }} <strong>${platformName}</strong> {{ __("account") }} <strong>${username}</strong>?</p>
                </div>
            `,
            icon: false,
            showCancelButton: true,
            confirmButtonColor: '#dc3545',
            cancelButtonColor: '#6c757d',
            confirmButtonText: '<i class="bx bx-x me-1"></i>{{ __("Yes, reject it!") }}',
            cancelButtonText: '{{ __("Cancel") }}'
        }).then((result) => {
            if (result.isConfirmed) {
                $.ajax({
                    url: `{{ url('admin/social-accounts') }}/${accountId}/reject`,
                    type: 'POST',
                    data: {
                        _token: '{{ csrf_token() }}'
                    },
                    success: function(response) {
                        if (response.success) {
                            Swal.fire({
                                title: '{{ __("Rejected!") }}',
                                text: response.message,
                                icon: 'success',
                                timer: 2000,
                                showConfirmButton: false
                            });
                            loadAccounts(currentPage);
                        } else {
                            showNotification('error', response.message);
                        }
                    },
                    error: function(xhr) {
                        const response = xhr.responseJSON;
                        showNotification('error', response?.message || '{{ __("An error occurred.") }}');
                    }
                });
            }
        });
    }

    function deleteAccount(accountId, platformName, username) {
        Swal.fire({
            title: '{{ __("Are you sure?") }}',
            html: `
                <div class="text-center">
                    <i class="bx bx-trash text-danger" style="font-size: 3rem;"></i>
                    <h5 class="mt-3">{{ __("Delete Account") }}</h5>
                    <p class="text-muted">{{ __("Do you want to delete") }} <strong>${platformName}</strong> {{ __("account") }} <strong>${username}</strong>?</p>
                    <p class="text-warning"><small>{{ __("This action cannot be undone.") }}</small></p>
                </div>
            `,
            icon: false,
            showCancelButton: true,
            confirmButtonColor: '#dc3545',
            cancelButtonColor: '#6c757d',
            confirmButtonText: '<i class="bx bx-trash me-1"></i>{{ __("Yes, delete it!") }}',
            cancelButtonText: '{{ __("Cancel") }}'
        }).then((result) => {
            if (result.isConfirmed) {
                $.ajax({
                    url: `{{ url('admin/social-accounts') }}/${accountId}`,
                    type: 'POST',
                    data: {
                        _token: '{{ csrf_token() }}',
                        _method: 'DELETE'
                    },
                    success: function(response) {
                        if (response.success) {
                            Swal.fire({
                                title: '{{ __("Deleted!") }}',
                                text: response.message,
                                icon: 'success',
                                timer: 2000,
                                showConfirmButton: false
                            });
                            loadAccounts(currentPage);
                        } else {
                            showNotification('error', response.message);
                        }
                    },
                    error: function(xhr) {
                        const response = xhr.responseJSON;
                        showNotification('error', response?.message || '{{ __("An error occurred.") }}');
                    }
                });
            }
        });
    }

    function showNotification(type, message) {
        Lobibox.notify(type, {
            pauseDelayOnHover: true,
            size: 'mini',
            rounded: true,
            delayIndicator: false,
            continueDelayOnInactiveTab: false,
            position: 'top right',
            msg: message
        });
    }
</script>
@endsection