@extends('layouts.app')

@section('style')
    <link href="{{ asset('assets/plugins/notifications/css/lobibox.min.css') }}" rel="stylesheet">
    <link href="{{ asset('assets/plugins/datatable/css/dataTables.bootstrap5.min.css') }}" rel="stylesheet">
    <style>
        .analytics-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border-radius: 20px;
            padding: 30px;
            margin-bottom: 30px;
            color: white;
            position: relative;
            overflow: hidden;
        }
        .analytics-header::before {
            content: '';
            position: absolute;
            top: 0;
            right: 0;
            width: 200px;
            height: 200px;
            background: rgba(255,255,255,0.1);
            border-radius: 50%;
            transform: translate(50px, -50px);
        }
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        .stat-card {
            background: white;
            border-radius: 16px;
            padding: 25px;
            box-shadow: 0 4px 20px rgba(0,0,0,0.08);
            border: 1px solid #f1f3f4;
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }
        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 8px 30px rgba(0,0,0,0.15);
        }
        .stat-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            width: 4px;
            height: 100%;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        }
        .stat-icon {
            width: 60px;
            height: 60px;
            border-radius: 16px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.8rem;
            margin-bottom: 15px;
        }
        .stat-number {
            font-size: 2.2rem;
            font-weight: 700;
            color: #2c3e50;
            margin-bottom: 5px;
        }
        .stat-label {
            color: #6c757d;
            font-size: 0.9rem;
            font-weight: 500;
        }
        .stat-change {
            font-size: 0.8rem;
            font-weight: 600;
            margin-top: 8px;
        }
        .stat-change.positive {
            color: #28a745;
        }
        .stat-change.negative {
            color: #dc3545;
        }
        
        /* Icon Colors */
        .stat-icon.primary { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; }
        .stat-icon.success { background: linear-gradient(135deg, #28a745 0%, #20c997 100%); color: white; }
        .stat-icon.warning { background: linear-gradient(135deg, #ffc107 0%, #fd7e14 100%); color: white; }
        .stat-icon.danger { background: linear-gradient(135deg, #dc3545 0%, #e83e8c 100%); color: white; }
        .stat-icon.info { background: linear-gradient(135deg, #17a2b8 0%, #6f42c1 100%); color: white; }
        
        .chart-container {
            background: white;
            border-radius: 16px;
            padding: 25px;
            box-shadow: 0 4px 20px rgba(0,0,0,0.08);
            margin-bottom: 30px;
        }
        .chart-header {
            display: flex;
            justify-content: between;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 15px;
            border-bottom: 2px solid #f8f9fa;
        }
        .chart-title {
            font-size: 1.2rem;
            font-weight: 600;
            color: #2c3e50;
        }
        
        .activity-item {
            display: flex;
            align-items: center;
            padding: 15px;
            border-radius: 12px;
            margin-bottom: 10px;
            background: #f8f9fa;
            transition: all 0.3s ease;
        }
        .activity-item:hover {
            background: #e9ecef;
            transform: translateX(5px);
        }
        .activity-avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: bold;
            margin-right: 15px;
        }
        .activity-content {
            flex: 1;
        }
        .activity-title {
            font-weight: 600;
            color: #2c3e50;
            margin-bottom: 2px;
        }
        .activity-meta {
            font-size: 0.85rem;
            color: #6c757d;
        }
        .activity-amount {
            font-weight: 700;
            color: #28a745;
        }
        
        .top-creators {
            background: white;
            border-radius: 16px;
            padding: 25px;
            box-shadow: 0 4px 20px rgba(0,0,0,0.08);
        }
        .creator-item {
            display: flex;
            align-items: center;
            justify-content: between;
            padding: 12px 0;
            border-bottom: 1px solid #f1f3f4;
        }
        .creator-item:last-child {
            border-bottom: none;
        }
        .creator-rank {
            width: 30px;
            height: 30px;
            border-radius: 50%;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: bold;
            font-size: 0.9rem;
            margin-right: 15px;
        }
        
        /* Mobile Responsive */
        @media (max-width: 768px) {
            .analytics-header {
                padding: 20px;
                text-align: center;
            }
            .stats-grid {
                grid-template-columns: 1fr;
                gap: 15px;
            }
            .stat-card {
                padding: 20px;
            }
            .stat-number {
                font-size: 1.8rem;
            }
            .chart-container {
                padding: 20px;
            }
            .activity-item {
                flex-direction: column;
                text-align: center;
            }
            .activity-avatar {
                margin-right: 0;
                margin-bottom: 10px;
            }
        }
        
        @media (max-width: 576px) {
            .analytics-header {
                padding: 15px;
            }
            .stat-card {
                padding: 15px;
            }
            .stat-number {
                font-size: 1.5rem;
            }
            .chart-container {
                padding: 15px;
            }
        }
        
        /* DataTable Responsive */
        .table-responsive {
            border-radius: 16px;
            box-shadow: 0 4px 20px rgba(0,0,0,0.08);
        }
        .table {
            margin-bottom: 0;
        }
        .table thead th {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border: none;
            font-weight: 600;
            padding: 15px;
        }
        .table tbody td {
            padding: 15px;
            border-color: #f1f3f4;
            vertical-align: middle;
        }
        .table tbody tr:hover {
            background: #f8f9fa;
        }
    </style>
    
    .table-responsive {
        border-radius: 10px;
        overflow: hidden;
    }
    
    .btn-action {
        margin: 2px;
        padding: 5px 10px;
        font-size: 12px;
    }
</style>
@endsection

@section('wrapper')
<div class="page-wrapper">
    <div class="page-content">
        <!-- Breadcrumb -->
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-3">
            <div class="breadcrumb-title pe-3">{{__('Admin Panel')}}</div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ route('dashboard') }}">{{__('Home')}}</a></li>
                        <li class="breadcrumb-item active" aria-current="page">{{__('Watch Tasks Analytics')}}</li>
                    </ol>
                </nav>
            </div>
        </div>

        <!-- Header -->
        <div class="admin-watch-container">
            <div class="row align-items-center">
                <div class="col-md-8">
                    <h3 class="mb-3">
                        <i class="bx bx-bar-chart-alt-2 me-2"></i>
                        {{__('Watch Tasks Analytics')}}
                    </h3>
                    <p class="text-muted mb-0">
                        {{__('Monitor and manage all YouTube watch tasks across the platform')}}
                    </p>
                </div>
                <div class="col-md-4 text-end">
                    <button class="btn btn-primary" onclick="refreshData()">
                        <i class="bx bx-refresh me-2"></i>{{__('Refresh Data')}}
                    </button>
                </div>
            </div>
        </div>

        <!-- Statistics Cards -->
        <div class="row">
            <div class="col-xl-3 col-md-6">
                <div class="stats-card">
                    <div class="stats-number">{{ $stats['total_tasks'] }}</div>
                    <div class="stats-label">{{__('Total Tasks')}}</div>
                </div>
            </div>
            <div class="col-xl-3 col-md-6">
                <div class="stats-card">
                    <div class="stats-number">{{ $stats['active_tasks'] }}</div>
                    <div class="stats-label">{{__('Active Tasks')}}</div>
                </div>
            </div>
            <div class="col-xl-3 col-md-6">
                <div class="stats-card">
                    <div class="stats-number">{{ $stats['total_completions'] }}</div>
                    <div class="stats-label">{{__('Total Completions')}}</div>
                </div>
            </div>
            <div class="col-xl-3 col-md-6">
                <div class="stats-card">
                    <div class="stats-number">{{ number_format($stats['total_rewards_paid'], 2) }}₺</div>
                    <div class="stats-label">{{__('Total Rewards Paid')}}</div>
                </div>
            </div>
        </div>

        <!-- Tasks Table -->
        <div class="admin-watch-container">
            <div class="d-flex justify-content-between align-items-center mb-4">
                <h5 class="mb-0">
                    <i class="bx bx-list-ul me-2"></i>
                    {{__('All Watch Tasks')}}
                </h5>
                <div class="d-flex gap-2">
                    <select class="form-select form-select-sm" id="statusFilter" style="width: auto;">
                        <option value="">{{__('All Status')}}</option>
                        <option value="1">{{__('Active')}}</option>
                        <option value="0">{{__('Inactive')}}</option>
                    </select>
                </div>
            </div>
            
            <div class="table-responsive">
                <table class="table table-striped" id="watchTasksTable">
                    <thead class="table-dark">
                        <tr>
                            <th>{{__('ID')}}</th>
                            <th>{{__('Title')}}</th>
                            <th>{{__('Creator')}}</th>
                            <th>{{__('Duration')}}</th>
                            <th>{{__('Reward')}}</th>
                            <th>{{__('Completions')}}</th>
                            <th>{{__('Status')}}</th>
                            <th>{{__('Created')}}</th>
                            <th>{{__('Actions')}}</th>
                        </tr>
                    </thead>
                    <tbody>
                        <!-- DataTable will populate this -->
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>

<!-- Task Details Modal -->
<div class="modal fade" id="taskDetailsModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">{{__('Task Details')}}</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body" id="taskDetailsContent">
                <!-- Content will be loaded here -->
            </div>
        </div>
    </div>
</div>
@endsection

@section('script')
<script>
$(document).ready(function() {
    initializeDataTable();
    
    $('#statusFilter').on('change', function() {
        table.ajax.reload();
    });
});

let table;

function initializeDataTable() {
    table = $('#watchTasksTable').DataTable({
        processing: true,
        serverSide: true,
        ajax: {
            url: '{{ route("api.admin.watch.datatable") }}',
            data: function(d) {
                d.status = $('#statusFilter').val();
            }
        },
        columns: [
            { data: 'id', name: 'id' },
            { data: 'title', name: 'title' },
            { data: 'creator', name: 'creator.account_holder' },
            { data: 'watch_duration', name: 'watch_duration' },
            { data: 'reward_amount', name: 'reward_amount' },
            { 
                data: 'completions', 
                name: 'completions',
                render: function(data, type, row) {
                    return `${row.current_completions}/${row.max_completions}`;
                }
            },
            { 
                data: 'is_active', 
                name: 'is_active',
                render: function(data, type, row) {
                    return data ? 
                        '<span class="badge bg-success">{{__("Active")}}</span>' : 
                        '<span class="badge bg-danger">{{__("Inactive")}}</span>';
                }
            },
            { data: 'created_at', name: 'created_at' },
            { 
                data: 'actions', 
                name: 'actions', 
                orderable: false, 
                searchable: false,
                render: function(data, type, row) {
                    return getActionButtons(row);
                }
            }
        ],
        order: [[0, 'desc']],
        language: {
            url: '//cdn.datatables.net/plug-ins/1.11.5/i18n/Turkish.json'
        }
    });
}

function getActionButtons(task) {
    let buttons = `
        <button class="btn btn-info btn-action" onclick="viewTask(${task.id})">
            <i class="bx bx-show"></i> {{__('View')}}
        </button>
    `;
    
    if (task.is_active) {
        buttons += `
            <button class="btn btn-warning btn-action" onclick="toggleTaskStatus(${task.id}, false)">
                <i class="bx bx-pause"></i> {{__('Deactivate')}}
            </button>
        `;
    } else {
        buttons += `
            <button class="btn btn-success btn-action" onclick="toggleTaskStatus(${task.id}, true)">
                <i class="bx bx-play"></i> {{__('Activate')}}
            </button>
        `;
    }
    
    buttons += `
        <button class="btn btn-danger btn-action" onclick="deleteTask(${task.id})">
            <i class="bx bx-trash"></i> {{__('Delete')}}
        </button>
    `;
    
    return buttons;
}

function viewTask(taskId) {
    // Load task details via AJAX
    $.get(`/api/admin/watch-tasks/${taskId}`, function(response) {
        if (response.success) {
            const task = response.task;
            const completions = response.completions;
            
            let content = `
                <div class="row">
                    <div class="col-md-6">
                        <h6>{{__('Basic Information')}}</h6>
                        <table class="table table-sm">
                            <tr><td><strong>{{__('Title')}}:</strong></td><td>${task.title}</td></tr>
                            <tr><td><strong>{{__('Description')}}:</strong></td><td>${task.description || '{{__("No description")}}'}</td></tr>
                            <tr><td><strong>{{__('Duration')}}:</strong></td><td>${task.formatted_duration}</td></tr>
                            <tr><td><strong>{{__('Reward')}}:</strong></td><td>${task.reward_amount}₺</td></tr>
                            <tr><td><strong>{{__('Max Completions')}}:</strong></td><td>${task.max_completions}</td></tr>
                            <tr><td><strong>{{__('Current Completions')}}:</strong></td><td>${task.current_completions}</td></tr>
                        </table>
                    </div>
                    <div class="col-md-6">
                        <h6>{{__('Video Preview')}}</h6>
                        <div class="ratio ratio-16x9">
                            <iframe src="https://www.youtube.com/embed/${task.youtube_video_id}" allowfullscreen></iframe>
                        </div>
                    </div>
                </div>
                
                <hr>
                
                <h6>{{__('Recent Completions')}}</h6>
                <div class="table-responsive">
                    <table class="table table-sm">
                        <thead>
                            <tr>
                                <th>{{__('User')}}</th>
                                <th>{{__('Watched Duration')}}</th>
                                <th>{{__('Earned Amount')}}</th>
                                <th>{{__('Completed At')}}</th>
                            </tr>
                        </thead>
                        <tbody>
            `;
            
            completions.forEach(completion => {
                content += `
                    <tr>
                        <td>${completion.user.account_holder}</td>
                        <td>${Math.floor(completion.watched_duration / 60)}:${(completion.watched_duration % 60).toString().padStart(2, '0')}</td>
                        <td>${completion.earned_amount}₺</td>
                        <td>${new Date(completion.completed_at).toLocaleString()}</td>
                    </tr>
                `;
            });
            
            content += `
                        </tbody>
                    </table>
                </div>
            `;
            
            $('#taskDetailsContent').html(content);
            $('#taskDetailsModal').modal('show');
        }
    });
}

function toggleTaskStatus(taskId, activate) {
    const action = activate ? '{{__("activate")}}' : '{{__("deactivate")}}';
    
    Swal.fire({
        title: '{{__("Are you sure?")}}',
        text: `{{__("Do you want to")}} ${action} {{__("this task?")}}`,
        icon: 'question',
        showCancelButton: true,
        confirmButtonText: '{{__("Yes")}}',
        cancelButtonText: '{{__("Cancel")}}'
    }).then((result) => {
        if (result.isConfirmed) {
            $.ajax({
                url: `/api/watch-tasks/${taskId}`,
                method: 'PATCH',
                data: {
                    _token: $('meta[name="csrf-token"]').attr('content')
                },
                success: function(response) {
                    if (response.success) {
                        table.ajax.reload();
                        Swal.fire('{{__("Success")}}', response.message, 'success');
                    }
                },
                error: function() {
                    Swal.fire('{{__("Error")}}', '{{__("An error occurred")}}', 'error');
                }
            });
        }
    });
}

function deleteTask(taskId) {
    Swal.fire({
        title: '{{__("Are you sure?")}}',
        text: '{{__("This action cannot be undone! All related data will be permanently deleted.")}}',
        icon: 'warning',
        showCancelButton: true,
        confirmButtonColor: '#d33',
        cancelButtonColor: '#3085d6',
        confirmButtonText: '{{__("Yes, delete it!")}}',
        cancelButtonText: '{{__("Cancel")}}'
    }).then((result) => {
        if (result.isConfirmed) {
            $.ajax({
                url: `/api/watch-tasks/${taskId}`,
                method: 'DELETE',
                data: {
                    _token: $('meta[name="csrf-token"]').attr('content')
                },
                success: function(response) {
                    if (response.success) {
                        table.ajax.reload();
                        Swal.fire('{{__("Deleted!")}}', response.message, 'success');
                    }
                },
                error: function() {
                    Swal.fire('{{__("Error")}}', '{{__("An error occurred")}}', 'error');
                }
            });
        }
    });
}

function refreshData() {
    table.ajax.reload();
    
    // Refresh statistics
    location.reload();
}
</script>
@endsection