@extends('layouts.app')

@section('title', __('Website Visit Proofs - Admin'))

@section('style')
    <link href="{{ asset('assets/plugins/notifications/css/lobibox.min.css') }}" rel="stylesheet">
    <link href="{{ asset('assets/plugins/datatable/css/dataTables.bootstrap5.min.css') }}" rel="stylesheet">
@endsection

@section('wrapper')
<div class="page-wrapper">
    <div class="page-content">
        <!--breadcrumb-->
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-3">
            <div class="breadcrumb-title pe-3">{{ __('Website Visit Proofs') }}</div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ route('dashboard') }}"><i class="bx bx-home-alt"></i></a></li>
                        <li class="breadcrumb-item"><a href="{{ route('admin.website-visits.index') }}">{{ __('Admin Panel') }}</a></li>
                        <li class="breadcrumb-item active" aria-current="page">{{ __('Proofs') }}</li>
                    </ol>
                </nav>
            </div>
            <div class="ms-auto">
                <a href="{{ route('admin.website-visits.index') }}" class="btn btn-secondary">
                    <i class='bx bx-arrow-back me-1'></i>{{ __('Back to Admin Panel') }}
                </a>
            </div>
        </div>
        <!--end breadcrumb-->

        <div class="card">
            <div class="card-body">
                @if($proofs->count() > 0)
                    <div class="table-responsive">
                        <table id="proofsTable" class="table table-striped table-bordered">
                            <thead>
                                <tr>
                                    <th>{{ __('User') }}</th>
                                    <th>{{ __('Task') }}</th>
                                    <th>{{ __('Description') }}</th>
                                    <th>{{ __('Files') }}</th>
                                    <th>{{ __('Status') }}</th>
                                    <th>{{ __('Submitted') }}</th>
                                    <th>{{ __('Actions') }}</th>
                                </tr>
                            </thead>
                            <tbody>
                                @foreach($proofs as $proof)
                                <tr>
                                    <td>
                                        <div class="d-flex align-items-center">
                                            <img src="{{ asset('assets/images/avatars/' . ($proof->user->avatar ?? 'avatar-1.png')) }}" 
                                                 alt="{{ $proof->user->account_holder ?? $proof->user->name }}" 
                                                 class="rounded-circle" width="32" height="32">
                                            <div class="ms-2">
                                                <a href="{{ route('user.profile') }}?user={{ $proof->user->id }}" class="text-decoration-none">
                                                    <h6 class="mb-0 text-primary">{{ $proof->user->account_holder ?? $proof->user->name }}</h6>
                                                </a>
                                                <p class="text-muted mb-0 small">{{ $proof->user->email }}</p>
                                            </div>
                                        </div>
                                    </td>
                                    <td>
                                        <div>
                                            <h6 class="mb-0">{{ $proof->visit->task->title }}</h6>
                                            <small class="text-muted">{{ $proof->visit->task->website_url }}</small>
                                        </div>
                                    </td>
                                    <td>
                                        <div style="max-width: 200px;">
                                            {{ Str::limit($proof->description, 100) }}
                                        </div>
                                    </td>
                                    <td>
                                        <div class="d-flex flex-wrap gap-1">
                                            @foreach($proof->files as $file)
                                                <button type="button" class="btn btn-sm btn-outline-primary" 
                                                        onclick="showImageOffcanvas('{{ $file->getUrl() }}', '{{ addslashes($proof->user->account_holder ?? $proof->user->name) }}', '{{ addslashes($proof->visit->task->title) }}')">
                                                    <i class='bx bx-image me-1'></i>{{ __('View') }}
                                                </button>
                                            @endforeach
                                        </div>
                                    </td>
                                    <td>
                                        @if($proof->status == 0)
                                            <span class="badge bg-warning">{{ __('Pending') }}</span>
                                        @elseif($proof->status == 1)
                                            <span class="badge bg-success">{{ __('Approved') }}</span>
                                        @else
                                            <span class="badge bg-danger">{{ __('Denied') }}</span>
                                        @endif
                                    </td>
                                    <td>{{ $proof->created_at->format('d/m/Y H:i') }}</td>
                                    <td>
                                        @if($proof->status == 0)
                                            <button class="btn btn-success btn-sm approve-btn" data-proof-id="{{ $proof->id }}" onclick="approveProof({{ $proof->id }})">
                                                <i class='bx bx-check me-1'></i>{{ __('Approve') }}
                                            </button>
                                            <button class="btn btn-danger btn-sm deny-btn" data-proof-id="{{ $proof->id }}" onclick="showDenyModal({{ $proof->id }})">
                                                <i class='bx bx-x me-1'></i>{{ __('Deny') }}
                                            </button>
                                        @else
                                            <span class="text-muted">{{ __('Reviewed') }}</span>
                                            @if($proof->reviewed_by)
                                                <br><small>{{ __('by') }} {{ $proof->reviewer->name ?? 'Unknown' }}</small>
                                            @endif
                                        @endif
                                    </td>
                                </tr>
                                @endforeach
                            </tbody>
                        </table>
                    </div>
                    
                    <div class="d-flex justify-content-center mt-3">
                        {{ $proofs->links() }}
                    </div>
                @else
                    <div class="text-center py-5">
                        <i class='bx bx-camera display-1 text-muted'></i>
                        <h4 class="mt-3">{{ __('No Pending Proofs') }}</h4>
                        <p class="text-muted">{{ __('There are no website visit proofs waiting for review.') }}</p>
                    </div>
                @endif
            </div>
        </div>
    </div>
</div>

<!-- Image Preview Offcanvas -->
<div class="offcanvas offcanvas-end" tabindex="-1" id="imagePreviewOffcanvas" aria-labelledby="imagePreviewOffcanvasLabel">
    <div class="offcanvas-header">
        <h5 class="offcanvas-title" id="imagePreviewOffcanvasLabel">{{ __('Proof Image') }}</h5>
        <button type="button" class="btn-close" data-bs-dismiss="offcanvas" aria-label="Close"></button>
    </div>
    <div class="offcanvas-body p-0">
        <!-- User and Task Info -->
        <div class="bg-light p-3 border-bottom">
            <div class="mb-2">
                <small class="text-muted">{{ __('User') }}:</small>
                <div id="offcanvasUserName" class="fw-bold"></div>
            </div>
            <div>
                <small class="text-muted">{{ __('Task') }}:</small>
                <div id="offcanvasTaskTitle" class="fw-bold"></div>
            </div>
        </div>
        
        <!-- Image Container -->
        <div class="p-3 text-center">
            <img id="offcanvasImage" src="" alt="{{ __('Proof Image') }}" class="img-fluid rounded" style="max-width: 100%; height: auto;">
        </div>
        
        <!-- Action Buttons -->
        <div class="p-3 border-top">
            <div class="d-grid gap-2">
                <a id="offcanvasDownloadBtn" href="" download class="btn btn-primary">
                    <i class='bx bx-download me-2'></i>{{ __('Download Image') }}
                </a>
                <a id="offcanvasOpenBtn" href="" target="_blank" class="btn btn-outline-info">
                    <i class='bx bx-link-external me-2'></i>{{ __('Open in New Tab') }}
                </a>
                <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="offcanvas">
                    <i class='bx bx-x me-2'></i>{{ __('Close') }}
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Deny Reason Modal -->
<div class="modal fade" id="denyReasonModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">{{ __('Deny Proof') }}</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <form id="denyForm">
                    <input type="hidden" id="denyProofId" name="proof_id">
                    <div class="mb-3">
                        <label class="form-label">{{ __('Reason for denial') }}</label>
                        <textarea name="reason" class="form-control" rows="3" required 
                            placeholder="{{ __('Please provide a reason for denying this proof...') }}"></textarea>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">{{ __('Cancel') }}</button>
                <button type="button" class="btn btn-danger" onclick="submitDeny()">{{ __('Deny Proof') }}</button>
            </div>
        </div>
    </div>
</div>
@endsection

@section('script')
    <script src="{{ asset('assets/plugins/notifications/js/lobibox.min.js') }}"></script>
    <script src="{{ asset('assets/plugins/notifications/js/notifications.min.js') }}"></script>
    <script src="{{ asset('assets/plugins/datatable/js/jquery.dataTables.min.js') }}"></script>
    <script src="{{ asset('assets/plugins/datatable/js/dataTables.bootstrap5.min.js') }}"></script>

    <script>
        $(document).ready(function() {
            $('#proofsTable').DataTable({
                order: [[5, 'desc']], // Sort by submitted date
                pageLength: 25
            });
        });

        // Show image in offcanvas
        function showImageOffcanvas(imageUrl, userName, taskTitle) {
            $('#offcanvasImage').attr('src', imageUrl);
            $('#offcanvasUserName').text(userName);
            $('#offcanvasTaskTitle').text(taskTitle);
            $('#offcanvasDownloadBtn').attr('href', imageUrl);
            $('#offcanvasOpenBtn').attr('href', imageUrl);
            
            // Show offcanvas
            const offcanvas = new bootstrap.Offcanvas(document.getElementById('imagePreviewOffcanvas'));
            offcanvas.show();
        }

        // AJAX Approve Proof
        function approveProof(proofId) {
            Swal.fire({
                title: '{{ __("Are you sure?") }}',
                text: '{{ __("This will approve the proof and give reward to user") }}',
                icon: 'question',
                showCancelButton: true,
                confirmButtonColor: '#28a745',
                cancelButtonColor: '#6c757d',
                confirmButtonText: '{{ __("Yes, approve it!") }}',
                cancelButtonText: '{{ __("Cancel") }}'
            }).then((result) => {
                if (result.isConfirmed) {
                    // Update button state
                    const button = document.querySelector(`button[data-proof-id="${proofId}"].approve-btn`);
                    const originalHtml = button.innerHTML;
                    
                    button.disabled = true;
                    button.innerHTML = '<i class="bx bx-loader-alt bx-spin me-1"></i>{{ __("Approving...") }}';
                    
                    // Make AJAX request
                    fetch(`{{ url('/admin/website-visit-proofs') }}/${proofId}/approve`, {
                        method: 'POST',
                        headers: {
                            'X-CSRF-TOKEN': '{{ csrf_token() }}',
                            'Content-Type': 'application/json',
                            'Accept': 'application/json'
                        }
                    })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            // Update button to approved state
                            button.innerHTML = '<i class="bx bx-check me-1"></i>{{ __("Approved") }}';
                            button.className = 'btn btn-success btn-sm';
                            
                            // Remove the row from table with animation
                            const row = button.closest('tr');
                            row.style.transition = 'opacity 0.5s ease-out';
                            row.style.opacity = '0';
                            
                            setTimeout(() => {
                                row.remove();
                                
                                // Check if table is empty
                                const tbody = document.querySelector('#proofsTable tbody');
                                if (tbody.children.length === 0) {
                                    tbody.innerHTML = '<tr><td colspan="6" class="text-center">{{ __("No Pending Proofs") }}</td></tr>';
                                }
                            }, 500);
                            
                            // Show success notification
                            Lobibox.notify('success', {
                                pauseDelayOnHover: true,
                                continueDelayOnInactiveTab: false,
                                position: 'top right',
                                icon: 'bx bx-check',
                                title: '{{ __("Success") }}',
                                msg: '{{ __("Proof approved successfully!") }}',
                                delay: 3000
                            });
                        } else {
                            throw new Error(data.message || 'Unknown error');
                        }
                    })
                    .catch(error => {
                        console.error('Error:', error);
                        
                        // Restore button state
                        button.disabled = false;
                        button.innerHTML = originalHtml;
                        
                        // Show error notification
                        Lobibox.notify('error', {
                            pauseDelayOnHover: true,
                            continueDelayOnInactiveTab: false,
                            position: 'top right',
                            icon: 'bx bx-x',
                            title: '{{ __("Error") }}',
                            msg: '{{ __("Failed to approve proof!") }}',
                            delay: 5000
                        });
                    });
                }
            });
        }

        function approveProof(proofId) {
            if (confirm('{{ __("Are you sure you want to approve this proof?") }}')) {
                $.ajax({
                    url: `{{ url('/admin/website-visit-proofs') }}/${proofId}/approve`,
                    type: 'POST',
                    headers: {
                        'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content'),
                        'X-Requested-With': 'XMLHttpRequest'
                    },
                    success: function(response) {
                        if (response.success) {
                            showNotification('success', '{{ __("Success") }}', response.message);
                            location.reload();
                        } else {
                            showNotification('error', '{{ __("Error") }}', response.message);
                        }
                    },
                    error: function(xhr) {
                        const message = xhr.responseJSON?.message || '{{ __("An error occurred") }}';
                        showNotification('error', '{{ __("Error") }}', message);
                    }
                });
            }
        }

        function denyProof(proofId) {
            $('#denyProofId').val(proofId);
            $('#denyReasonModal').modal('show');
        }

        function submitDeny() {
            const proofId = $('#denyProofId').val();
            const reason = $('#denyForm textarea[name="reason"]').val();

            if (!reason.trim()) {
                alert('{{ __("Please provide a reason for denial") }}');
                return;
            }

            $.ajax({
                url: `{{ url('/admin/website-visit-proofs') }}/${proofId}/deny`,
                type: 'POST',
                headers: {
                    'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content'),
                    'X-Requested-With': 'XMLHttpRequest'
                },
                data: {
                    reason: reason
                },
                success: function(response) {
                    if (response.success) {
                        showNotification('success', '{{ __("Success") }}', response.message);
                        $('#denyReasonModal').modal('hide');
                        location.reload();
                    } else {
                        showNotification('error', '{{ __("Error") }}', response.message);
                    }
                },
                error: function(xhr) {
                    const message = xhr.responseJSON?.message || '{{ __("An error occurred") }}';
                    showNotification('error', '{{ __("Error") }}', message);
                }
            });
        }

        function showNotification(type, title, message) {
            Lobibox.notify(type, {
                pauseDelayOnHover: true,
                continueDelayOnInactiveTab: false,
                position: 'top right',
                icon: type === 'success' ? 'bx bx-check-circle' : 'bx bx-x-circle',
                title: title,
                msg: message
            });
        }
    </script>
@endsection