@extends('layouts.app')

@section('style')
    <link href="{{ asset('assets/plugins/notifications/css/lobibox.min.css') }}" rel="stylesheet">
    <link href="{{ asset('assets/plugins/datatable/css/dataTables.bootstrap5.min.css') }}" rel="stylesheet">
    <style>
        .withdrawal-header {
            background: linear-gradient(135deg, #dc3545 0%, #fd7e14 100%);
            border-radius: 20px;
            padding: 30px;
            margin-bottom: 30px;
            color: white;
            position: relative;
            overflow: hidden;
        }
        .withdrawal-header::before {
            content: '';
            position: absolute;
            top: 0;
            right: 0;
            width: 200px;
            height: 200px;
            background: rgba(255,255,255,0.1);
            border-radius: 50%;
            transform: translate(50px, -50px);
        }
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        .stat-card {
            background: white;
            border-radius: 16px;
            padding: 25px;
            box-shadow: 0 4px 20px rgba(0,0,0,0.08);
            border: 1px solid #f1f3f4;
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }
        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 8px 30px rgba(0,0,0,0.15);
        }
        .stat-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            width: 4px;
            height: 100%;
            background: linear-gradient(135deg, #dc3545 0%, #fd7e14 100%);
        }
        .stat-icon {
            width: 60px;
            height: 60px;
            border-radius: 16px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.8rem;
            margin-bottom: 15px;
        }
        .stat-number {
            font-size: 2.2rem;
            font-weight: 700;
            color: #2c3e50;
            margin-bottom: 5px;
        }
        .stat-label {
            color: #6c757d;
            font-size: 0.9rem;
            font-weight: 500;
        }
        .stat-change {
            font-size: 0.8rem;
            font-weight: 600;
            margin-top: 8px;
        }
        .stat-change.positive {
            color: #28a745;
        }
        .stat-change.negative {
            color: #dc3545;
        }
        .stat-change.neutral {
            color: #6c757d;
        }
        
        /* Icon Colors */
        .stat-icon.primary { background: linear-gradient(135deg, #dc3545 0%, #fd7e14 100%); color: white; }
        .stat-icon.success { background: linear-gradient(135deg, #28a745 0%, #20c997 100%); color: white; }
        .stat-icon.warning { background: linear-gradient(135deg, #ffc107 0%, #fd7e14 100%); color: white; }
        .stat-icon.danger { background: linear-gradient(135deg, #dc3545 0%, #e83e8c 100%); color: white; }
        .stat-icon.info { background: linear-gradient(135deg, #17a2b8 0%, #6f42c1 100%); color: white; }
        
        .table-container {
            background: white;
            border-radius: 16px;
            padding: 25px;
            box-shadow: 0 4px 20px rgba(0,0,0,0.08);
            margin-bottom: 30px;
        }
        .table-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 15px;
            border-bottom: 2px solid #f8f9fa;
        }
        .table-title {
            font-size: 1.2rem;
            font-weight: 600;
            color: #2c3e50;
        }
        
        .status-badge {
            padding: 8px 16px;
            border-radius: 20px;
            font-weight: 600;
            font-size: 0.85rem;
        }
        .status-pending {
            background: linear-gradient(135deg, #ffc107 0%, #fd7e14 100%);
            color: white;
        }
        .status-approved {
            background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
            color: white;
        }
        .status-rejected {
            background: linear-gradient(135deg, #dc3545 0%, #e83e8c 100%);
            color: white;
        }
        .status-completed {
            background: linear-gradient(135deg, #17a2b8 0%, #6f42c1 100%);
            color: white;
        }
        
        .action-buttons {
            display: flex;
            gap: 5px;
            flex-wrap: wrap;
        }
        .action-buttons .btn {
            padding: 6px 12px;
            font-size: 0.8rem;
            border-radius: 8px;
            font-weight: 600;
        }
        
        .user-info {
            display: flex;
            align-items: center;
            gap: 10px;
            position: relative;
        }
        
        .user-actions {
            margin-left: auto;
            padding-left: 10px;
        }
        
        .user-actions .dropdown-toggle {
            border: none;
            background: transparent;
            color: #6c757d;
            padding: 4px 8px;
            border-radius: 50%;
            transition: all 0.2s ease;
        }
        
        .user-actions .dropdown-toggle:hover {
            background-color: #f8f9fa;
            color: #007bff;
            transform: scale(1.1);
        }
        
        .user-actions .dropdown-toggle:focus {
            box-shadow: 0 0 0 0.2rem rgba(0, 123, 255, 0.25);
        }
        
        .user-actions .dropdown-menu {
            border: none;
            box-shadow: 0 4px 20px rgba(0,0,0,0.15);
            border-radius: 8px;
            padding: 8px 0;
        }
        
        .user-actions .dropdown-item {
            padding: 8px 16px;
            transition: all 0.2s ease;
        }
        
        .user-actions .dropdown-item:hover {
            background-color: #f8f9fa;
            transform: translateX(2px);
        }
        
        /* Mobile Cards Styles */
        .mobile-withdrawal-card {
            background: white;
            border-radius: 12px;
            padding: 16px;
            margin-bottom: 12px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
            border: 1px solid #e9ecef;
            transition: all 0.2s ease;
        }
        
        .mobile-withdrawal-card:hover {
            box-shadow: 0 4px 16px rgba(0,0,0,0.15);
            transform: translateY(-1px);
        }
        
        .mobile-card-header {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            margin-bottom: 12px;
        }
        
        .mobile-user-info {
            flex: 1;
        }
        
        .mobile-user-name {
            font-weight: 600;
            color: #2c3e50;
            font-size: 1rem;
            margin-bottom: 2px;
        }
        
        .mobile-user-email {
            font-size: 0.85rem;
            color: #6c757d;
        }
        
        .mobile-amount {
            font-size: 1.2rem;
            font-weight: 700;
            color: #dc3545;
            text-align: right;
        }
        
        .mobile-card-body {
            border-top: 1px solid #f1f3f4;
            padding-top: 12px;
        }
        
        .mobile-info-row {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 8px;
        }
        
        .mobile-info-row:last-child {
            margin-bottom: 0;
        }
        
        .mobile-info-label {
            font-size: 0.85rem;
            color: #6c757d;
            font-weight: 500;
        }
        
        .mobile-info-value {
            font-size: 0.9rem;
            color: #2c3e50;
            font-weight: 600;
            text-align: right;
        }
        
        .mobile-iban {
            font-family: 'Courier New', monospace;
            background-color: #f8f9fa;
            padding: 2px 6px;
            border-radius: 4px;
            font-size: 0.8rem;
        }
        
        .mobile-status-badge {
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 0.75rem;
            font-weight: 600;
            text-transform: uppercase;
        }
        
        .mobile-actions {
            margin-top: 12px;
            padding-top: 12px;
            border-top: 1px solid #f1f3f4;
            display: flex;
            gap: 8px;
            flex-wrap: wrap;
        }
        
        .mobile-action-btn {
            flex: 1;
            min-width: 80px;
            padding: 8px 12px;
            border-radius: 8px;
            font-size: 0.85rem;
            font-weight: 500;
            border: none;
            transition: all 0.2s ease;
        }
        
        .mobile-action-btn:hover {
            transform: translateY(-1px);
        }
        
        .mobile-loading {
            text-align: center;
            padding: 40px 20px;
            color: #6c757d;
        }
        
        .mobile-empty {
            text-align: center;
            padding: 60px 20px;
            color: #6c757d;
        }
        
        .mobile-empty i {
            font-size: 3rem;
            margin-bottom: 16px;
            opacity: 0.5;
        }
        .user-avatar {
            width: 35px;
            height: 35px;
            border-radius: 50%;
            background: linear-gradient(135deg, #dc3545 0%, #fd7e14 100%);
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: bold;
            font-size: 0.9rem;
        }
        .user-details {
            flex: 1;
        }
        .user-name {
            font-weight: 600;
            color: #2c3e50;
            margin-bottom: 2px;
        }
        .user-email {
            font-size: 0.85rem;
            color: #6c757d;
        }
        
        .amount-display {
            font-size: 1.1rem;
            font-weight: 700;
            color: #dc3545;
        }
        
        /* Mobile Responsive */
        @media (max-width: 768px) {
            .withdrawal-header {
                padding: 20px;
                text-align: center;
            }
            .stats-grid {
                grid-template-columns: 1fr;
                gap: 15px;
            }
            .stat-card {
                padding: 20px;
            }
            .stat-number {
                font-size: 1.8rem;
            }
            .table-container {
                padding: 20px;
            }
            .action-buttons {
                justify-content: center;
            }
            .user-info {
                flex-direction: column;
                text-align: center;
            }
            .user-avatar {
                margin: 0 auto;
            }
        }
        
        @media (max-width: 576px) {
            .withdrawal-header {
                padding: 15px;
            }
            .stat-card {
                padding: 15px;
            }
            .stat-number {
                font-size: 1.5rem;
            }
            .table-container {
                padding: 15px;
            }
        }
        
        /* DataTable Responsive */
        .table-responsive {
            border-radius: 16px;
            box-shadow: 0 4px 20px rgba(0,0,0,0.08);
        }
        .table {
            margin-bottom: 0;
        }
        .table thead th {
            background: linear-gradient(135deg, #dc3545 0%, #fd7e14 100%);
            color: white;
            border: none;
            font-weight: 600;
            padding: 15px;
        }
        .table tbody td {
            padding: 15px;
            border-color: #f1f3f4;
            vertical-align: middle;
        }
        .table tbody tr:hover {
            background: #f8f9fa;
        }
        
        /* Modal Styles */
        .modal-header {
            background: linear-gradient(135deg, #dc3545 0%, #fd7e14 100%);
            color: white;
            border-radius: 12px 12px 0 0;
        }
        .modal-content {
            border-radius: 12px;
            border: none;
            box-shadow: 0 10px 40px rgba(0,0,0,0.2);
        }
        .form-control:focus {
            border-color: #dc3545;
            box-shadow: 0 0 0 0.2rem rgba(220, 53, 69, 0.25);
        }
        
        /* Withdrawal Details Styles */
        .withdrawal-details {
            padding: 0;
        }
        
        .detail-section {
            border-bottom: 1px solid #e9ecef;
            padding: 20px;
        }
        
        .detail-section:last-child {
            border-bottom: none;
        }
        
        .section-header {
            display: flex;
            align-items: center;
            gap: 10px;
            margin-bottom: 15px;
            padding-bottom: 10px;
            border-bottom: 1px solid #f1f3f4;
        }
        
        .section-header i {
            font-size: 1.2rem;
        }
        
        .section-header h6 {
            margin: 0;
            font-weight: 600;
            color: #2c3e50;
        }
        
        .section-content {
            padding-left: 30px;
        }
        
        .info-row {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 8px 0;
            border-bottom: 1px solid #f8f9fa;
        }
        
        .info-row:last-child {
            border-bottom: none;
        }
        
        .info-row .label {
            font-weight: 500;
            color: #6c757d;
            min-width: 120px;
        }
        
        .info-row .value {
            font-weight: 600;
            color: #2c3e50;
            text-align: right;
            flex: 1;
        }
        
        .balance-amount {
            color: #28a745 !important;
            font-size: 1.1rem;
        }
        
        .amount-highlight {
            color: #dc3545 !important;
            font-size: 1.2rem;
            font-weight: 700;
        }
        
        .iban-code {
            background-color: #f8f9fa;
            color: #495057;
            padding: 4px 8px;
            border-radius: 4px;
            font-family: 'Courier New', monospace;
            font-size: 0.9rem;
        }
        
        .clickable-iban {
            cursor: pointer;
            transition: all 0.2s ease;
        }
        
        .clickable-iban:hover {
            background-color: #e9ecef;
            color: #212529;
        }
        
        .copy-icon {
            cursor: pointer;
            transition: all 0.2s ease;
            font-size: 1rem;
        }
        
        .copy-icon:hover {
            color: #007bff !important;
            transform: scale(1.1);
        }
        
        .admin-note {
            background-color: #f8f9fa;
            padding: 15px;
            border-radius: 8px;
            border-left: 4px solid #17a2b8;
            font-style: italic;
            color: #495057;
        }
    </style>
@endsection

@section('wrapper')
<div class="page-wrapper">
    <div class="page-content">
        <!--breadcrumb-->
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-3">
            <div class="breadcrumb-title pe-3">{{__('Withdrawal Management')}}</div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ route('dashboard') }}">{{__('Home')}}</a></li>
                        <li class="breadcrumb-item active" aria-current="page">{{__('Withdrawal Requests')}}</li>
                    </ol>
                </nav>
            </div>
            <div class="ms-auto">
                <button class="btn btn-primary" onclick="refreshData()">
                    <i class="bx bx-refresh me-2"></i>{{__('Refresh')}}
                </button>
            </div>
        </div>
        <!--end breadcrumb-->

        <!-- Withdrawal Header -->
        <div class="withdrawal-header">
            <div class="row align-items-center">
                <div class="col-md-8">
                    <h2 class="mb-2 text-white">{{__('Withdrawal Management')}}</h2>
                    <p class="mb-0 opacity-75">{{__('Manage and process user withdrawal requests')}}</p>
                </div>
                <div class="col-md-4 text-md-end">
                    <div class="d-flex align-items-center justify-content-md-end">
                        <i class="bx bx-money-withdraw" style="font-size: 3rem; opacity: 0.3;"></i>
                    </div>
                </div>
            </div>
        </div>

        <!-- Statistics Grid -->
        <div class="stats-grid">
            <div class="stat-card">
                <div class="d-flex align-items-center">
                    <div class="stat-icon warning me-3">
                        <i class="bx bx-time"></i>
                    </div>
                    <div class="flex-grow-1">
                        <div class="stat-number" id="pendingCount">-</div>
                        <div class="stat-label">{{__('Pending Requests')}}</div>
                        <div class="stat-change neutral">
                            <i class="bx bx-loader-alt"></i> {{__('Awaiting review')}}
                        </div>
                    </div>
                </div>
            </div>

            <div class="stat-card">
                <div class="d-flex align-items-center">
                    <div class="stat-icon success me-3">
                        <i class="bx bx-check"></i>
                    </div>
                    <div class="flex-grow-1">
                        <div class="stat-number" id="approvedCount">-</div>
                        <div class="stat-label">{{__('Approved Requests')}}</div>
                        <div class="stat-change positive">
                            <i class="bx bx-trending-up"></i> {{__('Ready for payment')}}
                        </div>
                    </div>
                </div>
            </div>

            <div class="stat-card">
                <div class="d-flex align-items-center">
                    <div class="stat-icon info me-3">
                        <i class="bx bx-check-double"></i>
                    </div>
                    <div class="flex-grow-1">
                        <div class="stat-number" id="completedCount">-</div>
                        <div class="stat-label">{{__('Completed Payments')}}</div>
                        <div class="stat-change positive">
                            <i class="bx bx-wallet"></i> {{__('Successfully paid')}}
                        </div>
                    </div>
                </div>
            </div>

            <div class="stat-card">
                <div class="d-flex align-items-center">
                    <div class="stat-icon danger me-3">
                        <i class="bx bx-x"></i>
                    </div>
                    <div class="flex-grow-1">
                        <div class="stat-number" id="rejectedCount">-</div>
                        <div class="stat-label">{{__('Rejected Requests')}}</div>
                        <div class="stat-change negative">
                            <i class="bx bx-trending-down"></i> {{__('Declined')}}
                        </div>
                    </div>
                </div>
            </div>

            <div class="stat-card">
                <div class="d-flex align-items-center">
                    <div class="stat-icon primary me-3">
                        <i class="bx bx-money"></i>
                    </div>
                    <div class="flex-grow-1">
                        <div class="stat-number" id="totalAmount">-</div>
                        <div class="stat-label">{{__('Total Amount')}}</div>
                        <div class="stat-change neutral">
                            <i class="bx bx-calculator"></i> {{__('All requests')}}
                        </div>
                    </div>
                </div>
            </div>

            <div class="stat-card">
                <div class="d-flex align-items-center">
                    <div class="stat-icon success me-3">
                        <i class="bx bx-transfer"></i>
                    </div>
                    <div class="flex-grow-1">
                        <div class="stat-number" id="paidAmount">-</div>
                        <div class="stat-label">{{__('Paid Amount')}}</div>
                        <div class="stat-change positive">
                            <i class="bx bx-check-circle"></i> {{__('Completed payments')}}
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Withdrawal Requests Table -->
        <!-- Desktop Table -->
        <div class="table-container d-none d-lg-block">
            <div class="table-header">
                <h5 class="table-title">{{__('Withdrawal Requests')}}</h5>
                <div class="d-flex gap-2">
                    <select class="form-select form-select-sm" id="statusFilter" style="width: auto;">
                        <option value="">{{__('All Status')}}</option>
                        <option value="pending">{{__('Pending')}}</option>
                        <option value="approved">{{__('Approved')}}</option>
                        <option value="completed">{{__('Completed')}}</option>
                        <option value="rejected">{{__('Rejected')}}</option>
                    </select>
                </div>
            </div>
            
            <div class="table-responsive">
                <table id="withdrawalsTable" class="table table-striped">
                    <thead>
                        <tr>
                            <th>{{__('User')}}</th>
                            <th>{{__('Amount')}}</th>
                            <th>{{__('IBAN')}}</th>
                            <th>{{__('Account Holder')}}</th>
                            <th>{{__('Status')}}</th>
                            <th>{{__('Request Date')}}</th>
                        </tr>
                    </thead>
                    <tbody>
                        <!-- DataTable will populate this -->
                    </tbody>
                </table>
            </div>
        </div>

        <!-- Mobile Cards -->
        <div class="mobile-withdrawals d-lg-none">
            <div class="d-flex justify-content-between align-items-center mb-3">
                <h5 class="mb-0">{{__('Withdrawal Requests')}}</h5>
                <div class="d-flex gap-2">
                    <select class="form-select form-select-sm" id="statusFilterMobile" style="width: auto;">
                        <option value="">{{__('All Status')}}</option>
                        <option value="pending">{{__('Pending')}}</option>
                        <option value="approved">{{__('Approved')}}</option>
                        <option value="completed">{{__('Completed')}}</option>
                        <option value="rejected">{{__('Rejected')}}</option>
                    </select>
                    <button class="btn btn-sm btn-primary" onclick="refreshMobileData()">
                        <i class="bx bx-refresh"></i>
                    </button>
                </div>
            </div>
            
            <div id="mobileWithdrawalsContainer">
                <!-- Mobile cards will be loaded here -->
            </div>
            
            <div class="text-center mt-3">
                <button class="btn btn-outline-primary" id="loadMoreBtn" onclick="loadMoreMobileData()">
                    <i class="bx bx-loader-alt bx-spin d-none" id="loadMoreSpinner"></i>
                    {{__('Load More')}}
                </button>
            </div>
        </div>
    </div>
</div>

<!-- View Details Offcanvas -->
<div class="offcanvas offcanvas-end" tabindex="-1" id="viewOffcanvas" style="width: 500px;">
    <div class="offcanvas-header bg-primary text-white">
        <h5 class="offcanvas-title text-white">{{__('Withdrawal Details')}}</h5>
        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="offcanvas"></button>
    </div>
    <div class="offcanvas-body p-0">
        <div id="withdrawalDetails">
            <!-- Details will be loaded here -->
        </div>
    </div>
</div>

<!-- Approve Offcanvas -->
<div class="offcanvas offcanvas-end" tabindex="-1" id="approveOffcanvas" style="width: 400px;">
    <div class="offcanvas-header bg-success text-white">
        <h5 class="offcanvas-title text-white">{{__('Approve Withdrawal')}}</h5>
        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="offcanvas"></button>
    </div>
    <div class="offcanvas-body">
        <div class="alert alert-info">
            <i class="bx bx-info-circle me-2"></i>
            {{__('Are you sure you want to approve this withdrawal request?')}}
        </div>
        <div class="mb-3">
            <label class="form-label">{{__('Admin Note (Optional)')}}</label>
            <textarea class="form-control" id="approveNote" rows="3" placeholder="{{__('Add any notes for this approval...')}}"></textarea>
        </div>
        <div class="d-grid gap-2">
            <button type="button" class="btn btn-success btn-lg" onclick="processAction('approve')">
                <i class="bx bx-check me-2"></i>{{__('Approve Withdrawal')}}
            </button>
            <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="offcanvas">{{__('Cancel')}}</button>
        </div>
    </div>
</div>

<!-- Reject Offcanvas -->
<div class="offcanvas offcanvas-end" tabindex="-1" id="rejectOffcanvas" style="width: 400px;">
    <div class="offcanvas-header bg-danger text-white">
        <h5 class="offcanvas-title text-white">{{__('Reject Withdrawal')}}</h5>
        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="offcanvas"></button>
    </div>
    <div class="offcanvas-body">
        <div class="alert alert-warning">
            <i class="bx bx-error me-2"></i>
            {{__('Are you sure you want to reject this withdrawal request?')}}
        </div>
        <div class="mb-3">
            <label class="form-label">{{__('Rejection Reason')}} <span class="text-danger">*</span></label>
            <textarea class="form-control" id="rejectReason" rows="4" placeholder="{{__('Please provide a reason for rejection...')}}" required></textarea>
        </div>
        <div class="d-grid gap-2">
            <button type="button" class="btn btn-danger btn-lg" onclick="processAction('reject')">
                <i class="bx bx-x me-2"></i>{{__('Reject Withdrawal')}}
            </button>
            <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="offcanvas">{{__('Cancel')}}</button>
        </div>
    </div>
</div>

<!-- Complete Offcanvas -->
<div class="offcanvas offcanvas-end" tabindex="-1" id="completeOffcanvas" style="width: 400px;">
    <div class="offcanvas-header bg-primary text-white">
        <h5 class="offcanvas-title text-white">{{__('Complete Withdrawal')}}</h5>
        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="offcanvas"></button>
    </div>
    <div class="offcanvas-body">
        <div class="alert alert-success">
            <i class="bx bx-check-double me-2"></i>
            {{__('Are you sure you want to complete this withdrawal?')}}
        </div>
        <div class="mb-3">
            <label class="form-label">{{__('Transaction Reference')}} <span class="text-danger">*</span></label>
            <input type="text" class="form-control" id="transactionRef" placeholder="{{__('Enter bank transaction reference')}}" required>
            <div class="form-text">{{__('Enter the bank transaction reference number for tracking')}}</div>
        </div>
        <div class="d-grid gap-2">
            <button type="button" class="btn btn-primary btn-lg" onclick="processAction('complete')">
                <i class="bx bx-check-double me-2"></i>{{__('Complete Withdrawal')}}
            </button>
            <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="offcanvas">{{__('Cancel')}}</button>
        </div>
    </div>
</div>
@endsection

@section('script')
    <script src="{{ asset('assets/plugins/notifications/js/lobibox.min.js') }}"></script>
    <script src="{{ asset('assets/plugins/datatable/js/jquery.dataTables.min.js') }}"></script>
    <script src="{{ asset('assets/plugins/datatable/js/dataTables.bootstrap5.min.js') }}"></script>

    <script>
        let withdrawalsTable;
        let currentWithdrawalId = null;

        $(document).ready(function() {
            // Initialize DataTable
            withdrawalsTable = $('#withdrawalsTable').DataTable({
                processing: true,
                serverSide: true,
                ajax: {
                    url: '{{ route('api.admin.withdrawals.datatable') }}',
                    type: 'GET',
                    data: function(d) {
                        d.status = $('#statusFilter').val();
                    },
                    error: function(xhr, error, thrown) {
                        console.error('DataTable Ajax Error:', xhr.responseText);
                        showNotification('error', 'Veri yüklenirken hata oluştu: ' + xhr.status);
                    }
                },
                columns: [
                    { 
                        data: 'user_name', 
                        name: 'user_name',
                        render: function(data, type, row) {
                            return `
                                <div class="user-info">
                                    <div class="user-avatar">
                                        ${data.charAt(0).toUpperCase()}
                                    </div>
                                    <div class="user-details">
                                        <div class="user-name">${data}</div>
                                        <div class="user-email">${row.user_email}</div>
                                    </div>
                                    <div class="user-actions">
                                        <div class="dropdown">
                                            <button class="btn btn-sm btn-outline-primary dropdown-toggle" type="button" data-bs-toggle="dropdown" aria-expanded="false">
                                                <i class="bx bx-dots-vertical-rounded"></i>
                                            </button>
                                            <ul class="dropdown-menu">
                                                <li><a class="dropdown-item view-withdrawal" href="javascript:;" data-id="${row.id}">
                                                    <i class="bx bx-show me-2"></i>{{__('View Details')}}
                                                </a></li>
                                                ${row.status === 'pending' ? `
                                                <li><a class="dropdown-item approve-withdrawal" href="javascript:;" data-id="${row.id}">
                                                    <i class="bx bx-check me-2 text-success"></i>{{__('Approve')}}
                                                </a></li>
                                                <li><a class="dropdown-item reject-withdrawal" href="javascript:;" data-id="${row.id}">
                                                    <i class="bx bx-x me-2 text-danger"></i>{{__('Reject')}}
                                                </a></li>
                                                ` : row.status === 'approved' ? `
                                                <li><a class="dropdown-item complete-withdrawal" href="javascript:;" data-id="${row.id}">
                                                    <i class="bx bx-check-double me-2 text-primary"></i>{{__('Complete')}}
                                                </a></li>
                                                ` : ''}
                                            </ul>
                                        </div>
                                    </div>
                                </div>
                            `;
                        }
                    },
                    { 
                        data: 'amount', 
                        name: 'amount',
                        render: function(data) {
                            return `<div class="amount-display">${data}{{ $settings['currency'] ?? 'TL' }}</div>`;
                        }
                    },
                    { 
                        data: 'iban', 
                        name: 'iban',
                        render: function(data, type, row) {
                            return `<code class="iban-code clickable-iban" onclick="copyIban('${row.iban}')" title="{{__('Click to copy full IBAN')}}">${row.iban_masked || maskIban(row.iban)}</code>`;
                        }
                    },
                    { data: 'account_holder_name', name: 'account_holder_name' },
                    { 
                        data: 'status', 
                        name: 'status',
                        render: function(data, type, row) {
                            const statusClasses = {
                                'pending': 'status-pending',
                                'approved': 'status-approved',
                                'completed': 'status-completed',
                                'rejected': 'status-rejected'
                            };
                            const statusTexts = {
                                'pending': '{{__('Pending')}}',
                                'approved': '{{__('Approved')}}',
                                'completed': '{{__('Completed')}}',
                                'rejected': '{{__('Rejected')}}'
                            };
                            return `<span class="status-badge ${statusClasses[data]}">${statusTexts[data]}</span>`;
                        }
                    },
                    { data: 'created_at', name: 'created_at' }
                ],
                order: [[4, 'desc']], // Order by created_at desc (now column 4)
                pageLength: 25,
                responsive: true,
                language: {
                    search: "{{__('Search')}}:",
                    lengthMenu: "{{__('Show')}} _MENU_ {{__('entries')}}",
                    info: "{{__('Showing')}} _START_ {{__('to')}} _END_ {{__('of')}} _TOTAL_ {{__('entries')}}",
                    paginate: {
                        first: "{{__('First')}}",
                        last: "{{__('Last')}}",
                        next: "{{__('Next')}}",
                        previous: "{{__('Previous')}}"
                    }
                },
                drawCallback: function() {
                    updateStatistics();
                }
            });

            // Status filter change
            $('#statusFilter').on('change', function() {
                withdrawalsTable.ajax.reload();
            });

            // Load initial statistics
            updateStatistics();

            // Event handlers for action buttons
            $(document).on('click', '.view-withdrawal', function() {
                currentWithdrawalId = $(this).data('id');
                loadWithdrawalDetails(currentWithdrawalId);
                const offcanvas = new bootstrap.Offcanvas(document.getElementById('viewOffcanvas'));
                offcanvas.show();
            });

            $(document).on('click', '.approve-withdrawal', function() {
                currentWithdrawalId = $(this).data('id');
                $('#approveNote').val(''); // Clear previous input
                const offcanvas = new bootstrap.Offcanvas(document.getElementById('approveOffcanvas'));
                offcanvas.show();
            });

            $(document).on('click', '.reject-withdrawal', function() {
                currentWithdrawalId = $(this).data('id');
                $('#rejectReason').val(''); // Clear previous input
                const offcanvas = new bootstrap.Offcanvas(document.getElementById('rejectOffcanvas'));
                offcanvas.show();
            });

            $(document).on('click', '.complete-withdrawal', function() {
                currentWithdrawalId = $(this).data('id');
                $('#transactionRef').val(''); // Clear previous input
                const offcanvas = new bootstrap.Offcanvas(document.getElementById('completeOffcanvas'));
                offcanvas.show();
            });
        });

        function processAction(action) {
            const urls = {
                'approve': `{{ url('/admin/withdrawals') }}/${currentWithdrawalId}/approve`,
                'reject': `{{ url('/admin/withdrawals') }}/${currentWithdrawalId}/reject`,
                'complete': `{{ url('/admin/withdrawals') }}/${currentWithdrawalId}/complete`
            };

            let data = {
                _token: '{{ csrf_token() }}'
            };
            
            if (action === 'approve') {
                data.note = $('#approveNote').val();
            } else if (action === 'reject') {
                data.reason = $('#rejectReason').val();
                if (!data.reason.trim()) {
                    showNotification('error', '{{__('Please provide a reason for rejection')}}');
                    return;
                }
            } else if (action === 'complete') {
                data.transaction_reference = $('#transactionRef').val();
                if (!data.transaction_reference.trim()) {
                    showNotification('error', '{{__('Please provide transaction reference')}}');
                    return;
                }
            }

            // Disable button during processing
            const button = $(`#${action}Offcanvas .btn-${action === 'approve' ? 'success' : action === 'reject' ? 'danger' : 'primary'}`);
            const originalText = button.html();
            button.prop('disabled', true).html('<i class="bx bx-loader-alt bx-spin me-2"></i>{{__('Processing...')}}');

            $.post(urls[action], data)
                .done(function(response) {
                    if (response.success) {
                        showNotification('success', response.message);
                        // Hide offcanvas
                        const offcanvas = bootstrap.Offcanvas.getInstance(document.getElementById(`${action}Offcanvas`));
                        offcanvas.hide();
                        // Reload data
                        withdrawalsTable.ajax.reload(null, false); // Keep current page
                        updateStatistics();
                    } else {
                        showNotification('error', response.message);
                    }
                })
                .fail(function(xhr) {
                    let errorMessage = '{{__('Error processing request')}}';
                    if (xhr.responseJSON && xhr.responseJSON.message) {
                        errorMessage = xhr.responseJSON.message;
                    }
                    showNotification('error', errorMessage);
                })
                .always(function() {
                    // Re-enable button
                    button.prop('disabled', false).html(originalText);
                });
        }

        function loadWithdrawalDetails(withdrawalId) {
            $('#withdrawalDetails').html('<div class="text-center p-4"><i class="bx bx-loader-alt bx-spin fs-2"></i><br>{{__('Loading...')}}</div>');
            
            // Get withdrawal details from DataTable data
            const table = withdrawalsTable.rows().data();
            let withdrawal = null;
            
            for (let i = 0; i < table.length; i++) {
                if (table[i].id == withdrawalId) {
                    withdrawal = table[i];
                    break;
                }
            }
            
            if (!withdrawal) {
                $('#withdrawalDetails').html('<div class="text-center p-4 text-danger">{{__('Withdrawal not found')}}</div>');
                return;
            }
            
            // Fetch additional user details via AJAX
            $.get(`{{ url('/admin/withdrawals') }}/${withdrawalId}/details`)
                .done(function(response) {
                    if (response.success) {
                        const data = response.data;
                        renderWithdrawalDetails(data);
                    } else {
                        renderWithdrawalDetailsFromTable(withdrawal);
                    }
                })
                .fail(function() {
                    renderWithdrawalDetailsFromTable(withdrawal);
                });
        }
        
        function renderWithdrawalDetails(data) {
            const detailsHtml = `
                <div class="withdrawal-details">
                    <!-- User Information -->
                    <div class="detail-section">
                        <div class="section-header">
                            <i class="bx bx-user text-primary"></i>
                            <h6>{{__('User Information')}}</h6>
                        </div>
                        <div class="section-content">
                            <div class="info-row">
                                <span class="label">{{__('Name')}}:</span>
                                <span class="value">${data.user_name}</span>
                            </div>
                            <div class="info-row">
                                <span class="label">{{__('Email')}}:</span>
                                <span class="value">${data.user_email}</span>
                            </div>
                            <div class="info-row">
                                <span class="label">{{__('Phone')}}:</span>
                                <span class="value">${data.user_phone || 'N/A'}</span>
                            </div>
                            <div class="info-row">
                                <span class="label">{{__('Current Balance')}}:</span>
                                <span class="value balance-amount">${data.user_balance}{{ $settings['currency'] ?? 'TL' }}</span>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Withdrawal Information -->
                    <div class="detail-section">
                        <div class="section-header">
                            <i class="bx bx-money text-success"></i>
                            <h6>{{__('Withdrawal Information')}}</h6>
                        </div>
                        <div class="section-content">
                            <div class="info-row">
                                <span class="label">{{__('Withdrawal Amount')}}:</span>
                                <span class="value amount-highlight">${data.amount}{{ $settings['currency'] ?? 'TL' }}</span>
                            </div>
                            <div class="info-row">
                                <span class="label">{{__('IBAN')}}:</span>
                                <span class="value">
                                    <code class="iban-code clickable-iban" onclick="copyIban('${data.iban}')" title="{{__('Click to copy full IBAN')}}" data-full-iban="${data.iban}">
                                        ${maskIban(data.iban)}
                                    </code>
                                    <i class="bx bx-copy ms-2 text-muted copy-icon" onclick="copyIban('${data.iban}')" title="{{__('Copy IBAN')}}"></i>
                                </span>
                            </div>
                            <div class="info-row">
                                <span class="label">{{__('Account Holder')}}:</span>
                                <span class="value">${data.account_holder_name}</span>
                            </div>
                            <div class="info-row">
                                <span class="label">{{__('Status')}}:</span>
                                <span class="value"><span class="status-badge status-${data.status}">${data.status_text}</span></span>
                            </div>
                            <div class="info-row">
                                <span class="label">{{__('Request Date')}}:</span>
                                <span class="value">${data.created_at}</span>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Balance Information -->
                    <div class="detail-section">
                        <div class="section-header">
                            <i class="bx bx-wallet text-warning"></i>
                            <h6>{{__('Balance Information')}}</h6>
                        </div>
                        <div class="section-content">
                            <div class="info-row">
                                <span class="label">{{__('Balance Before')}}:</span>
                                <span class="value">${data.balance_before}{{ $settings['currency'] ?? 'TL' }}</span>
                            </div>
                            <div class="info-row">
                                <span class="label">{{__('Balance After')}}:</span>
                                <span class="value">${data.balance_after}{{ $settings['currency'] ?? 'TL' }}</span>
                            </div>
                        </div>
                    </div>
                    
                    ${data.admin_note ? `
                    <div class="detail-section">
                        <div class="section-header">
                            <i class="bx bx-note text-info"></i>
                            <h6>{{__('Admin Note')}}</h6>
                        </div>
                        <div class="section-content">
                            <div class="admin-note">${data.admin_note}</div>
                        </div>
                    </div>
                    ` : ''}
                    
                    ${data.transaction_reference ? `
                    <div class="detail-section">
                        <div class="section-header">
                            <i class="bx bx-receipt text-primary"></i>
                            <h6>{{__('Transaction Reference')}}</h6>
                        </div>
                        <div class="section-content">
                            <div class="info-row">
                                <span class="label">{{__('Reference')}}:</span>
                                <span class="value"><code>${data.transaction_reference}</code></span>
                            </div>
                        </div>
                    </div>
                    ` : ''}
                    
                    <!-- Action Buttons -->
                    ${data.status === 'pending' ? `
                    <div class="detail-section">
                        <div class="section-header">
                            <i class="bx bx-cog text-secondary"></i>
                            <h6>{{__('Actions')}}</h6>
                        </div>
                        <div class="section-content">
                            <div class="d-grid gap-2">
                                <button type="button" class="btn btn-success" onclick="approveFromDetails(${data.id})">
                                    <i class="bx bx-check me-2"></i>{{__('Approve Withdrawal')}}
                                </button>
                                <button type="button" class="btn btn-danger" onclick="rejectFromDetails(${data.id})">
                                    <i class="bx bx-x me-2"></i>{{__('Reject Withdrawal')}}
                                </button>
                            </div>
                        </div>
                    </div>
                    ` : data.status === 'approved' ? `
                    <div class="detail-section">
                        <div class="section-header">
                            <i class="bx bx-cog text-secondary"></i>
                            <h6>{{__('Actions')}}</h6>
                        </div>
                        <div class="section-content">
                            <div class="d-grid gap-2">
                                <button type="button" class="btn btn-primary" onclick="completeFromDetails(${data.id})">
                                    <i class="bx bx-check-double me-2"></i>{{__('Complete Withdrawal')}}
                                </button>
                            </div>
                        </div>
                    </div>
                    ` : ''}
                </div>
            `;
            $('#withdrawalDetails').html(detailsHtml);
        }
        
        function renderWithdrawalDetailsFromTable(withdrawal) {
            // Fallback to table data if AJAX fails
            const detailsHtml = `
                <div class="withdrawal-details">
                    <div class="detail-section">
                        <div class="section-header">
                            <i class="bx bx-user text-primary"></i>
                            <h6>{{__('User Information')}}</h6>
                        </div>
                        <div class="section-content">
                            <div class="info-row">
                                <span class="label">{{__('Name')}}:</span>
                                <span class="value">${withdrawal.user_name}</span>
                            </div>
                            <div class="info-row">
                                <span class="label">{{__('Email')}}:</span>
                                <span class="value">${withdrawal.user_email}</span>
                            </div>
                        </div>
                    </div>
                    
                    <div class="detail-section">
                        <div class="section-header">
                            <i class="bx bx-money text-success"></i>
                            <h6>{{__('Withdrawal Information')}}</h6>
                        </div>
                        <div class="section-content">
                            <div class="info-row">
                                <span class="label">{{__('Amount')}}:</span>
                                <span class="value amount-highlight">${withdrawal.amount}{{ $settings['currency'] ?? 'TL' }}</span>
                            </div>
                            <div class="info-row">
                                <span class="label">{{__('IBAN')}}:</span>
                                <span class="value">
                                    <code class="iban-code clickable-iban" onclick="copyIban('${withdrawal.iban}')" title="{{__('Click to copy full IBAN')}}" data-full-iban="${withdrawal.iban}">
                                        ${maskIban(withdrawal.iban)}
                                    </code>
                                    <i class="bx bx-copy ms-2 text-muted copy-icon" onclick="copyIban('${withdrawal.iban}')" title="{{__('Copy IBAN')}}"></i>
                                </span>
                            </div>
                            <div class="info-row">
                                <span class="label">{{__('Account Holder')}}:</span>
                                <span class="value">${withdrawal.account_holder_name}</span>
                            </div>
                            <div class="info-row">
                                <span class="label">{{__('Status')}}:</span>
                                <span class="value"><span class="status-badge status-${withdrawal.status}">${withdrawal.status_text}</span></span>
                            </div>
                        </div>
                    </div>
                    
                    ${withdrawal.status === 'pending' ? `
                    <div class="detail-section">
                        <div class="section-header">
                            <i class="bx bx-cog text-secondary"></i>
                            <h6>{{__('Actions')}}</h6>
                        </div>
                        <div class="section-content">
                            <div class="d-grid gap-2">
                                <button type="button" class="btn btn-success" onclick="approveFromDetails(${withdrawal.id})">
                                    <i class="bx bx-check me-2"></i>{{__('Approve Withdrawal')}}
                                </button>
                                <button type="button" class="btn btn-danger" onclick="rejectFromDetails(${withdrawal.id})">
                                    <i class="bx bx-x me-2"></i>{{__('Reject Withdrawal')}}
                                </button>
                            </div>
                        </div>
                    </div>
                    ` : withdrawal.status === 'approved' ? `
                    <div class="detail-section">
                        <div class="section-header">
                            <i class="bx bx-cog text-secondary"></i>
                            <h6>{{__('Actions')}}</h6>
                        </div>
                        <div class="section-content">
                            <div class="d-grid gap-2">
                                <button type="button" class="btn btn-primary" onclick="completeFromDetails(${withdrawal.id})">
                                    <i class="bx bx-check-double me-2"></i>{{__('Complete Withdrawal')}}
                                </button>
                            </div>
                        </div>
                    </div>
                    ` : ''}
                </div>
            `;
            $('#withdrawalDetails').html(detailsHtml);
        }
        
        // Action functions from details view
        function approveFromDetails(withdrawalId) {
            currentWithdrawalId = withdrawalId;
            $('#approveNote').val('');
            const viewOffcanvas = bootstrap.Offcanvas.getInstance(document.getElementById('viewOffcanvas'));
            viewOffcanvas.hide();
            setTimeout(() => {
                const approveOffcanvas = new bootstrap.Offcanvas(document.getElementById('approveOffcanvas'));
                approveOffcanvas.show();
            }, 300);
        }
        
        function rejectFromDetails(withdrawalId) {
            currentWithdrawalId = withdrawalId;
            $('#rejectReason').val('');
            const viewOffcanvas = bootstrap.Offcanvas.getInstance(document.getElementById('viewOffcanvas'));
            viewOffcanvas.hide();
            setTimeout(() => {
                const rejectOffcanvas = new bootstrap.Offcanvas(document.getElementById('rejectOffcanvas'));
                rejectOffcanvas.show();
            }, 300);
        }
        
        function completeFromDetails(withdrawalId) {
            currentWithdrawalId = withdrawalId;
            $('#transactionRef').val('');
            const viewOffcanvas = bootstrap.Offcanvas.getInstance(document.getElementById('viewOffcanvas'));
            viewOffcanvas.hide();
            setTimeout(() => {
                const completeOffcanvas = new bootstrap.Offcanvas(document.getElementById('completeOffcanvas'));
                completeOffcanvas.show();
            }, 300);
        }
        
        // IBAN masking function
        function maskIban(iban) {
            if (!iban || iban.length < 8) return iban;
            const start = iban.substring(0, 4);
            const end = iban.substring(iban.length - 4);
            const middle = '*'.repeat(iban.length - 8);
            return start + middle + end;
        }
        
        // Copy IBAN to clipboard
        function copyIban(iban) {
            if (navigator.clipboard && window.isSecureContext) {
                navigator.clipboard.writeText(iban).then(function() {
                    showNotification('success', '{{__('IBAN copied to clipboard')}}: ' + iban);
                }).catch(function() {
                    fallbackCopyTextToClipboard(iban);
                });
            } else {
                fallbackCopyTextToClipboard(iban);
            }
        }
        
        function fallbackCopyTextToClipboard(text) {
            const textArea = document.createElement("textarea");
            textArea.value = text;
            textArea.style.top = "0";
            textArea.style.left = "0";
            textArea.style.position = "fixed";
            document.body.appendChild(textArea);
            textArea.focus();
            textArea.select();
            
            try {
                const successful = document.execCommand('copy');
                if (successful) {
                    showNotification('success', '{{__('IBAN copied to clipboard')}}: ' + text);
                } else {
                    showNotification('error', '{{__('Failed to copy IBAN')}}');
                }
            } catch (err) {
                showNotification('error', '{{__('Failed to copy IBAN')}}');
            }
            
            document.body.removeChild(textArea);
        }

        function updateStatistics() {
            $.ajax({
                url: '{{ route('api.admin.withdrawals.datatable') }}',
                method: 'GET',
                data: { get_stats: true },
                success: function(response) {
                    if (response.stats) {
                        $('#pendingCount').text(response.stats.pending || 0);
                        $('#approvedCount').text(response.stats.approved || 0);
                        $('#completedCount').text(response.stats.completed || 0);
                        $('#rejectedCount').text(response.stats.rejected || 0);
                        $('#totalAmount').text((response.stats.total_amount || 0) + '{{ $settings['currency'] ?? 'TL' }}');
                        $('#paidAmount').text((response.stats.paid_amount || 0) + '{{ $settings['currency'] ?? 'TL' }}');
                    }
                }
            });
        }

        function refreshData() {
            withdrawalsTable.ajax.reload();
            updateStatistics();
            showNotification('info', '{{__('Data refreshed successfully')}}');
        }

        function showNotification(type, message) {
            Lobibox.notify(type, {
                pauseDelayOnHover: true,
                continueDelayOnInactiveTab: false,
                position: 'top right',
                icon: type === 'success' ? 'bx bx-check-circle' : 'bx bx-x-circle',
                msg: message
            });
        }

        // Mobile data variables
        let mobileCurrentPage = 0;
        let mobileHasMore = true;
        let mobileCurrentStatus = '';

        // Initialize mobile data on page load
        if (window.innerWidth < 992) {
            loadMobileData();
        }

        // Mobile status filter
        $('#statusFilterMobile').on('change', function() {
            mobileCurrentStatus = $(this).val();
            mobileCurrentPage = 0;
            mobileHasMore = true;
            loadMobileData(true);
        });

        function loadMobileData(reset = false) {
            if (reset) {
                $('#mobileWithdrawalsContainer').empty();
                mobileCurrentPage = 0;
            }

            $('#loadMoreSpinner').removeClass('d-none');
            $('#loadMoreBtn').prop('disabled', true);

            $.ajax({
                url: '{{ route('api.admin.withdrawals.datatable') }}',
                method: 'GET',
                data: {
                    start: mobileCurrentPage * 10,
                    length: 10,
                    status: mobileCurrentStatus
                },
                success: function(response) {
                    if (response.data && response.data.length > 0) {
                        response.data.forEach(function(withdrawal) {
                            const card = createMobileCard(withdrawal);
                            $('#mobileWithdrawalsContainer').append(card);
                        });
                        mobileCurrentPage++;
                        mobileHasMore = response.data.length === 10;
                    } else {
                        mobileHasMore = false;
                        if (mobileCurrentPage === 0) {
                            $('#mobileWithdrawalsContainer').html(`
                                <div class="mobile-empty">
                                    <i class="bx bx-inbox"></i>
                                    <p>{{__('No withdrawal requests found')}}</p>
                                </div>
                            `);
                        }
                    }
                    
                    $('#loadMoreBtn').toggle(mobileHasMore);
                },
                error: function() {
                    showNotification('error', '{{__('Error loading mobile data')}}');
                },
                complete: function() {
                    $('#loadMoreSpinner').addClass('d-none');
                    $('#loadMoreBtn').prop('disabled', false);
                }
            });
        }

        function createMobileCard(withdrawal) {
            const statusClasses = {
                'pending': 'bg-warning text-dark',
                'approved': 'bg-info text-white',
                'completed': 'bg-success text-white',
                'rejected': 'bg-danger text-white'
            };

            const statusTexts = {
                'pending': '{{__('Pending')}}',
                'approved': '{{__('Approved')}}',
                'completed': '{{__('Completed')}}',
                'rejected': '{{__('Rejected')}}'
            };

            let actionButtons = '';
            if (withdrawal.status === 'pending') {
                actionButtons = `
                    <button class="mobile-action-btn btn btn-success view-withdrawal" data-id="${withdrawal.id}">
                        <i class="bx bx-show me-1"></i>{{__('View')}}
                    </button>
                    <button class="mobile-action-btn btn btn-success approve-withdrawal" data-id="${withdrawal.id}">
                        <i class="bx bx-check me-1"></i>{{__('Approve')}}
                    </button>
                    <button class="mobile-action-btn btn btn-danger reject-withdrawal" data-id="${withdrawal.id}">
                        <i class="bx bx-x me-1"></i>{{__('Reject')}}
                    </button>
                `;
            } else if (withdrawal.status === 'approved') {
                actionButtons = `
                    <button class="mobile-action-btn btn btn-primary view-withdrawal" data-id="${withdrawal.id}">
                        <i class="bx bx-show me-1"></i>{{__('View')}}
                    </button>
                    <button class="mobile-action-btn btn btn-primary complete-withdrawal" data-id="${withdrawal.id}">
                        <i class="bx bx-check-double me-1"></i>{{__('Complete')}}
                    </button>
                `;
            } else {
                actionButtons = `
                    <button class="mobile-action-btn btn btn-primary view-withdrawal" data-id="${withdrawal.id}">
                        <i class="bx bx-show me-1"></i>{{__('View Details')}}
                    </button>
                `;
            }

            return `
                <div class="mobile-withdrawal-card">
                    <div class="mobile-card-header">
                        <div class="mobile-user-info">
                            <div class="mobile-user-name">${withdrawal.user_name}</div>
                            <div class="mobile-user-email">${withdrawal.user_email}</div>
                        </div>
                        <div class="mobile-amount">${withdrawal.amount}{{ $settings['currency'] ?? 'TL' }}</div>
                    </div>
                    
                    <div class="mobile-card-body">
                        <div class="mobile-info-row">
                            <span class="mobile-info-label">{{__('IBAN')}}:</span>
                            <span class="mobile-info-value">
                                <span class="mobile-iban clickable-iban" onclick="copyIban('${withdrawal.iban}')" title="{{__('Click to copy full IBAN')}}">
                                    ${withdrawal.iban_masked || maskIban(withdrawal.iban)}
                                </span>
                                <i class="bx bx-copy ms-1 text-muted copy-icon" onclick="copyIban('${withdrawal.iban}')" title="{{__('Copy IBAN')}}" style="font-size: 0.9rem;"></i>
                            </span>
                        </div>
                        <div class="mobile-info-row">
                            <span class="mobile-info-label">{{__('Account Holder')}}:</span>
                            <span class="mobile-info-value">${withdrawal.account_holder_name}</span>
                        </div>
                        <div class="mobile-info-row">
                            <span class="mobile-info-label">{{__('Status')}}:</span>
                            <span class="mobile-info-value">
                                <span class="mobile-status-badge ${statusClasses[withdrawal.status]}">${statusTexts[withdrawal.status]}</span>
                            </span>
                        </div>
                        <div class="mobile-info-row">
                            <span class="mobile-info-label">{{__('Date')}}:</span>
                            <span class="mobile-info-value">${withdrawal.created_at}</span>
                        </div>
                    </div>
                    
                    <div class="mobile-actions">
                        ${actionButtons}
                    </div>
                </div>
            `;
        }

        function loadMoreMobileData() {
            if (mobileHasMore) {
                loadMobileData();
            }
        }

        function refreshMobileData() {
            mobileCurrentPage = 0;
            mobileHasMore = true;
            loadMobileData(true);
            updateStatistics();
            showNotification('info', '{{__('Data refreshed successfully')}}');
        }

        // Auto refresh every 30 seconds
        setInterval(function() {
            if (document.hidden === false) {
                updateStatistics();
                // Auto refresh mobile data if on mobile
                if (window.innerWidth < 992 && mobileCurrentPage === 1) {
                    refreshMobileData();
                }
            }
        }, 30000);
    </script>
@endsection