@extends('gorevsitesi.layouts.app')

@section('title', 'Ürünler - GorevSitesi')
@section('description', 'GorevSitesi\'nde binlerce kaliteli ürün arasından size en uygun olanları keşfedin.')

@section('content')
<!-- Page Header -->
<div class="row mb-4">
    <div class="col-12">
        <div class="d-flex justify-content-between align-items-center">
            <div>
                <h1 class="fw-bold mb-2">Ürünler</h1>
                <p class="text-muted mb-0">Kaliteli ürünler, uygun fiyatlar</p>
            </div>
            <div class="d-flex gap-2">
                <button class="btn btn-outline-secondary" id="gridView" data-view="grid">
                    <i class="bi bi-grid"></i>
                </button>
                <button class="btn btn-outline-secondary" id="listView" data-view="list">
                    <i class="bi bi-list"></i>
                </button>
            </div>
        </div>
    </div>
</div>

<div class="row">
    <!-- Sidebar Filters -->
    <div class="col-lg-3 mb-4">
        <div class="card border-0 shadow-sm">
            <div class="card-header bg-white border-0">
                <h5 class="mb-0 fw-bold">
                    <i class="bi bi-funnel me-2"></i>Filtreler
                </h5>
            </div>
            <div class="card-body">
                <!-- Search -->
                <div class="mb-4">
                    <label class="form-label fw-medium">Ürün Ara</label>
                    <div class="input-group">
                        <input type="text" class="form-control" id="searchInput" placeholder="Ürün adı...">
                        <button class="btn btn-outline-secondary" type="button" id="searchBtn">
                            <i class="bi bi-search"></i>
                        </button>
                    </div>
                </div>

                <!-- Categories -->
                <div class="mb-4">
                    <label class="form-label fw-medium">Kategoriler</label>
                    <div class="list-group list-group-flush">
                        <a href="#" class="list-group-item list-group-item-action border-0 px-0 category-filter active" data-category="">
                            <i class="bi bi-grid me-2"></i>Tüm Kategoriler
                        </a>
                        @foreach($categories as $category)
                            <a href="#" class="list-group-item list-group-item-action border-0 px-0 category-filter" data-category="{{ $category->id }}">
                                <i class="bi bi-tag me-2"></i>{{ $category->name }}
                            </a>
                        @endforeach
                    </div>
                </div>

                <!-- Price Range -->
                <div class="mb-4">
                    <label class="form-label fw-medium">Fiyat Aralığı</label>
                    <div class="row g-2">
                        <div class="col-6">
                            <input type="number" class="form-control" id="minPrice" placeholder="Min">
                        </div>
                        <div class="col-6">
                            <input type="number" class="form-control" id="maxPrice" placeholder="Max">
                        </div>
                    </div>
                    <button class="btn btn-outline-primary btn-sm mt-2 w-100" id="applyPriceFilter">
                        Uygula
                    </button>
                </div>

                <!-- Sort Options -->
                <div class="mb-3">
                    <label class="form-label fw-medium">Sıralama</label>
                    <select class="form-select" id="sortSelect">
                        <option value="newest">En Yeni</option>
                        <option value="price_asc">Fiyat (Düşük → Yüksek)</option>
                        <option value="price_desc">Fiyat (Yüksek → Düşük)</option>
                        <option value="name_asc">İsim (A → Z)</option>
                        <option value="name_desc">İsim (Z → A)</option>
                    </select>
                </div>
            </div>
        </div>
    </div>

    <!-- Products Grid -->
    <div class="col-lg-9">
        <!-- Results Info -->
        <div class="d-flex justify-content-between align-items-center mb-3">
            <div>
                <span class="text-muted" id="resultsCount">{{ $products->count() }} ürün bulundu</span>
            </div>
            <div class="d-flex align-items-center gap-2">
                <span class="text-muted small">Sayfa başına:</span>
                <select class="form-select form-select-sm" id="perPageSelect" style="width: auto;">
                    <option value="12">12</option>
                    <option value="24">24</option>
                    <option value="48">48</option>
                </select>
            </div>
        </div>

        <!-- Products Container -->
        <div class="row g-4" id="products-container">
            @foreach($products as $product)
                <div class="col-lg-4 col-md-6 product-item" data-category="{{ $product->category_id ?? '' }}" data-price="{{ $product->price }}" data-name="{{ strtolower($product->product_name) }}">
                    <div class="card product-card h-100 border-0 shadow-sm">
                        <div class="position-relative">
                            <img src="{{ $product->image_url ?? '/assets/images/placeholder.png' }}" 
                                 class="card-img-top product-image" 
                                 alt="{{ $product->product_name }}"
                                 loading="lazy">
                            
                            @if($product->is_digital)
                                <span class="badge bg-info position-absolute top-0 start-0 m-2">
                                    <i class="bi bi-download me-1"></i>Dijital
                                </span>
                            @endif
                            
                            @if(isset($product->stock_quantity) && $product->stock_quantity <= 5 && !$product->is_digital)
                                <span class="badge bg-warning position-absolute top-0 end-0 m-2">
                                    Son {{ $product->stock_quantity }} adet
                                </span>
                            @endif
                        </div>
                        
                        <div class="card-body d-flex flex-column">
                            <h5 class="card-title text-truncate-2 mb-2">{{ $product->product_name }}</h5>
                            <p class="card-text text-muted small text-truncate-3 flex-grow-1">
                                {{ $product->description }}
                            </p>
                            
                            <div class="mt-auto">
                                <div class="d-flex justify-content-between align-items-center mb-3">
                                    <div>
                                        <span class="h5 fw-bold text-primary mb-0">{{ number_format($product->price, 2) }} ₺</span>
                                        @if($product->category)
                                            <div class="small text-muted">{{ $product->category->name }}</div>
                                        @endif
                                    </div>
                                    <div class="text-end">
                                        @if($product->is_digital)
                                            <span class="badge bg-info">Dijital Ürün</span>
                                        @elseif(isset($product->stock_quantity))
                                            @if($product->stock_quantity > 10)
                                                <span class="badge bg-success">Stokta</span>
                                            @elseif($product->stock_quantity > 0)
                                                <span class="badge bg-warning">Az Stok</span>
                                            @else
                                                <span class="badge bg-danger">Stokta Yok</span>
                                            @endif
                                        @else
                                            <span class="badge bg-success">Mevcut</span>
                                        @endif
                                    </div>
                                </div>
                                
                                <div class="d-grid gap-2">
                                    <a href="{{ route('site2.products.show', $product->id) }}" 
                                       class="btn btn-outline-primary">
                                        <i class="bi bi-eye me-2"></i>İncele
                                    </a>
                                    @php
                                        $isAvailable = $product->status && ($product->is_digital || ($product->stock_quantity ?? 0) > 0);
                                    @endphp
                                    @if($isAvailable)
                                        <button class="btn btn-primary add-to-cart" 
                                                data-product-id="{{ $product->id }}"
                                                data-product-name="{{ $product->product_name }}"
                                                data-product-price="{{ $product->price }}"
                                                data-product-image="{{ $product->image_url ?? '/assets/images/placeholder.png' }}"
                                                data-max-quantity="{{ $product->max_quantity ?? 100 }}"
                                                data-min-quantity="{{ $product->min_quantity ?? 1 }}">
                                            <i class="bi bi-bag-plus me-2"></i>Sepete Ekle
                                        </button>
                                    @else
                                        <button class="btn btn-secondary" disabled>
                                            <i class="bi bi-x-circle me-2"></i>Stokta Yok
                                        </button>
                                    @endif
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            @endforeach
        </div>

        <!-- Loading Spinner -->
        <div class="text-center py-5 d-none" id="loadingSpinner">
            <div class="spinner-border text-primary" role="status">
                <span class="visually-hidden">Yükleniyor...</span>
            </div>
            <p class="mt-2 text-muted">Ürünler yükleniyor...</p>
        </div>

        <!-- No Results -->
        <div class="text-center py-5 d-none" id="noResults">
            <i class="bi bi-search text-muted" style="font-size: 3rem;"></i>
            <h4 class="mt-3">Ürün bulunamadı</h4>
            <p class="text-muted">Arama kriterlerinizi değiştirip tekrar deneyin.</p>
        </div>

        <!-- Pagination -->
        <div class="d-flex justify-content-center mt-4" id="pagination">
            <!-- Pagination will be handled by JavaScript -->
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script>
    // Product filtering and search functionality
    document.addEventListener('DOMContentLoaded', function() {
        const productsContainer = document.getElementById('products-container');
        const searchInput = document.getElementById('searchInput');
        const categoryFilters = document.querySelectorAll('.category-filter');
        const sortSelect = document.getElementById('sortSelect');
        const minPriceInput = document.getElementById('minPrice');
        const maxPriceInput = document.getElementById('maxPrice');
        const applyPriceBtn = document.getElementById('applyPriceFilter');
        const resultsCount = document.getElementById('resultsCount');
        
        let currentFilters = {
            search: '',
            category: '',
            minPrice: '',
            maxPrice: '',
            sort: 'newest'
        };

        // Search functionality
        searchInput.addEventListener('input', debounce(function() {
            currentFilters.search = this.value.toLowerCase();
            filterProducts();
        }, 300));

        // Category filtering
        categoryFilters.forEach(filter => {
            filter.addEventListener('click', function(e) {
                e.preventDefault();
                categoryFilters.forEach(f => f.classList.remove('active'));
                this.classList.add('active');
                currentFilters.category = this.dataset.category;
                filterProducts();
            });
        });

        // Sort functionality
        sortSelect.addEventListener('change', function() {
            currentFilters.sort = this.value;
            filterProducts();
        });

        // Price filter
        applyPriceBtn.addEventListener('click', function() {
            currentFilters.minPrice = minPriceInput.value;
            currentFilters.maxPrice = maxPriceInput.value;
            filterProducts();
        });

        // Add to cart functionality
        document.addEventListener('click', function(e) {
            if (e.target.classList.contains('add-to-cart') || e.target.closest('.add-to-cart')) {
                const btn = e.target.classList.contains('add-to-cart') ? e.target : e.target.closest('.add-to-cart');
                addToCart(btn);
            }
        });

        function filterProducts() {
            const products = document.querySelectorAll('.product-item');
            let visibleCount = 0;

            products.forEach(product => {
                let visible = true;

                // Search filter
                if (currentFilters.search && !product.dataset.name.includes(currentFilters.search)) {
                    visible = false;
                }

                // Category filter
                if (currentFilters.category && product.dataset.category !== currentFilters.category) {
                    visible = false;
                }

                // Price filter
                const price = parseFloat(product.dataset.price);
                if (currentFilters.minPrice && price < parseFloat(currentFilters.minPrice)) {
                    visible = false;
                }
                if (currentFilters.maxPrice && price > parseFloat(currentFilters.maxPrice)) {
                    visible = false;
                }

                if (visible) {
                    product.style.display = 'block';
                    visibleCount++;
                } else {
                    product.style.display = 'none';
                }
            });

            // Update results count
            resultsCount.textContent = `${visibleCount} ürün bulundu`;

            // Show/hide no results message
            const noResults = document.getElementById('noResults');
            if (visibleCount === 0) {
                noResults.classList.remove('d-none');
                productsContainer.classList.add('d-none');
            } else {
                noResults.classList.add('d-none');
                productsContainer.classList.remove('d-none');
            }

            // Sort products
            sortProducts();
        }

        function sortProducts() {
            const products = Array.from(document.querySelectorAll('.product-item:not([style*="display: none"])'));
            
            products.sort((a, b) => {
                switch (currentFilters.sort) {
                    case 'price_asc':
                        return parseFloat(a.dataset.price) - parseFloat(b.dataset.price);
                    case 'price_desc':
                        return parseFloat(b.dataset.price) - parseFloat(a.dataset.price);
                    case 'name_asc':
                        return a.dataset.name.localeCompare(b.dataset.name);
                    case 'name_desc':
                        return b.dataset.name.localeCompare(a.dataset.name);
                    default:
                        return 0;
                }
            });

            // Reorder products in DOM
            products.forEach(product => {
                productsContainer.appendChild(product);
            });
        }

        function addToCart(btn) {
            const productData = {
                id: btn.dataset.productId,
                name: btn.dataset.productName,
                price: parseFloat(btn.dataset.productPrice),
                image: btn.dataset.productImage,
                maxQuantity: parseInt(btn.dataset.maxQuantity),
                minQuantity: parseInt(btn.dataset.minQuantity),
                quantity: 1
            };

            // Get existing cart
            let cart = JSON.parse(localStorage.getItem('cart') || '[]');
            
            // Check if product already in cart
            const existingIndex = cart.findIndex(item => item.id === productData.id);
            
            if (existingIndex > -1) {
                cart[existingIndex].quantity += 1;
                if (cart[existingIndex].quantity > productData.maxQuantity) {
                    cart[existingIndex].quantity = productData.maxQuantity;
                    alert(`Maksimum ${productData.maxQuantity} adet ekleyebilirsiniz.`);
                }
            } else {
                cart.push(productData);
            }

            // Save cart
            localStorage.setItem('cart', JSON.stringify(cart));
            
            // Update cart count
            updateCartCount(cart.length);
            
            // Show success message
            const originalText = btn.innerHTML;
            btn.innerHTML = '<i class="bi bi-check me-2"></i>Eklendi!';
            btn.classList.remove('btn-primary');
            btn.classList.add('btn-success');
            
            setTimeout(() => {
                btn.innerHTML = originalText;
                btn.classList.remove('btn-success');
                btn.classList.add('btn-primary');
            }, 1500);
        }

        function debounce(func, wait) {
            let timeout;
            return function executedFunction(...args) {
                const later = () => {
                    clearTimeout(timeout);
                    func(...args);
                };
                clearTimeout(timeout);
                timeout = setTimeout(later, wait);
            };
        }
    });
</script>
@endpush