@extends('layouts.app')

@section('title', __('Notifications'))

@section('style')
<style>
    .notifications-header {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        border-radius: 20px;
        padding: 30px;
        margin-bottom: 30px;
        color: white;
        position: relative;
        overflow: hidden;
    }
    .notifications-header::before {
        content: '';
        position: absolute;
        top: 0;
        right: 0;
        width: 200px;
        height: 200px;
        background: rgba(255,255,255,0.1);
        border-radius: 50%;
        transform: translate(50px, -50px);
    }
    .notification-card {
        background: white;
        border-radius: 16px;
        padding: 20px;
        margin-bottom: 15px;
        box-shadow: 0 4px 20px rgba(0,0,0,0.08);
        border: 1px solid #f1f3f4;
        transition: all 0.3s ease;
        position: relative;
        overflow: hidden;
    }
    .notification-card:hover {
        transform: translateY(-2px);
        box-shadow: 0 8px 30px rgba(0,0,0,0.15);
    }
    .notification-card.unread {
        border-left: 4px solid #007bff;
        background: linear-gradient(135deg, #f8f9ff 0%, #ffffff 100%);
    }
    .notification-card.read {
        border-left: 4px solid transparent;
    }
    .notification-icon {
        width: 50px;
        height: 50px;
        border-radius: 16px;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 1.5rem;
        margin-right: 15px;
        flex-shrink: 0;
    }
    .notification-content {
        flex: 1;
        min-width: 0;
    }
    .notification-title {
        font-size: 1rem;
        font-weight: 600;
        color: #2c3e50;
        margin-bottom: 5px;
        line-height: 1.4;
    }
    .notification-desc {
        color: #6c757d;
        font-size: 0.9rem;
        line-height: 1.5;
        margin-bottom: 10px;
        word-wrap: break-word;
        overflow-wrap: break-word;
    }
    .notification-time {
        font-size: 0.8rem;
        color: #9ca3af;
        display: flex;
        align-items: center;
    }
    .notification-actions {
        display: flex;
        gap: 8px;
        margin-top: 10px;
    }
    .btn-modern {
        border-radius: 8px;
        padding: 6px 12px;
        font-size: 0.8rem;
        font-weight: 500;
        transition: all 0.3s ease;
        border: none;
    }
    .btn-modern:hover {
        transform: translateY(-1px);
        box-shadow: 0 4px 12px rgba(0,0,0,0.15);
    }
    .stats-card {
        background: white;
        border-radius: 16px;
        padding: 20px;
        box-shadow: 0 4px 20px rgba(0,0,0,0.08);
        border: 1px solid #f1f3f4;
        text-align: center;
        transition: all 0.3s ease;
    }
    .stats-card:hover {
        transform: translateY(-2px);
        box-shadow: 0 8px 30px rgba(0,0,0,0.15);
    }
    .stats-number {
        font-size: 2rem;
        font-weight: 700;
        color: #2c3e50;
        margin-bottom: 5px;
    }
    .stats-label {
        color: #6c757d;
        font-size: 0.9rem;
        font-weight: 500;
    }
    .filter-tabs {
        background: white;
        border-radius: 16px;
        padding: 8px;
        margin-bottom: 20px;
        box-shadow: 0 4px 20px rgba(0,0,0,0.08);
    }
    .filter-tab {
        padding: 10px 20px;
        border-radius: 12px;
        border: none;
        background: transparent;
        color: #6c757d;
        font-weight: 500;
        transition: all 0.3s ease;
        margin: 0 4px;
    }
    .filter-tab.active {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        box-shadow: 0 4px 12px rgba(102, 126, 234, 0.3);
    }
    .empty-state {
        text-align: center;
        padding: 60px 20px;
        color: #6c757d;
    }
    .empty-icon {
        width: 80px;
        height: 80px;
        border-radius: 50%;
        background: linear-gradient(135deg, #f1f3f4 0%, #e9ecef 100%);
        display: flex;
        align-items: center;
        justify-content: center;
        margin: 0 auto 20px;
        font-size: 2rem;
        color: #9ca3af;
    }
    @media (max-width: 768px) {
        .notifications-header {
            padding: 20px;
            margin-bottom: 20px;
        }
        .notification-card {
            padding: 15px;
            margin-bottom: 10px;
        }
        .notification-icon {
            width: 40px;
            height: 40px;
            font-size: 1.2rem;
            margin-right: 12px;
        }
        .filter-tabs {
            padding: 6px;
        }
        .filter-tab {
            padding: 8px 16px;
            font-size: 0.9rem;
        }
    }
</style>
@endsection

@section('wrapper')
<div class="page-wrapper">
    <div class="page-content">
        <!--breadcrumb-->
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-3">
            <div class="breadcrumb-title pe-3">{{__('Notifications')}}</div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ route('dashboard') }}">{{__('Home')}}</a></li>
                        <li class="breadcrumb-item active" aria-current="page">{{__('Notifications')}}</li>
                    </ol>
                </nav>
            </div>
            <div class="ms-auto">
                <button class="btn btn-primary" onclick="markAllAsRead()">
                    <i class="bx bx-check-double me-2"></i>{{__('Mark All as Read')}}
                </button>
            </div>
        </div>
        <!--end breadcrumb-->

        <!-- Notifications Header -->
        <div class="notifications-header">
            <div class="row align-items-center">
                <div class="col-md-8">
                    <h2 class="mb-2 text-white">{{__('Your Notifications')}}</h2>
                    <p class="mb-0 opacity-75">{{__('Stay updated with all your important notifications')}}</p>
                </div>
                <div class="col-md-4 text-md-end">
                    <div class="d-flex align-items-center justify-content-md-end">
                        <i class="bx bx-bell" style="font-size: 3rem; opacity: 0.3;"></i>
                    </div>
                </div>
            </div>
        </div>

        <!-- Statistics -->
        <div class="row mb-4">
            <div class="col-lg-3 col-md-6 mb-3">
                <div class="stats-card">
                    <div class="stats-number text-primary" id="totalCount">{{ $notifications->count() }}</div>
                    <div class="stats-label">{{__('Total Notifications')}}</div>
                </div>
            </div>
            <div class="col-lg-3 col-md-6 mb-3">
                <div class="stats-card">
                    <div class="stats-number text-warning" id="unreadCountStat">{{ $unreadCount }}</div>
                    <div class="stats-label">{{__('Unread')}}</div>
                </div>
            </div>
            <div class="col-lg-3 col-md-6 mb-3">
                <div class="stats-card">
                    <div class="stats-number text-success" id="readCount">{{ $notifications->count() - $unreadCount }}</div>
                    <div class="stats-label">{{__('Read')}}</div>
                </div>
            </div>
            <div class="col-lg-3 col-md-6 mb-3">
                <div class="stats-card">
                    <div class="stats-number text-info" id="todayCount">{{ $notifications->where('created_at', '>=', today())->count() }}</div>
                    <div class="stats-label">{{__('Today')}}</div>
                </div>
            </div>
        </div>

        <!-- Filter Tabs -->
        <div class="filter-tabs">
            <div class="d-flex flex-wrap justify-content-center">
                <button class="filter-tab active" data-filter="all">{{__('All')}}</button>
                <button class="filter-tab" data-filter="unread">{{__('Unread')}}</button>
                <button class="filter-tab" data-filter="task_created">{{__('Tasks')}}</button>
                <button class="filter-tab" data-filter="ticket_created">{{__('Tickets')}}</button>
                <button class="filter-tab" data-filter="withdrawal_request">{{__('Withdrawals')}}</button>
                <button class="filter-tab" data-filter="announcement">{{__('Announcements')}}</button>
            </div>
        </div>

        <!-- Notifications List -->
        <div id="notificationsContainer">
            @forelse($notifications as $notification)
                @php
                    $iconClass = 'bx-bell';
                    $iconColor = 'text-primary';
                    $bgColor = 'rgba(var(--bs-primary-rgb), 0.1)';
                    
                    switch($notification->type) {
                        case 'task_created':
                        case 'task_approved':
                        case 'new_content':
                            $iconClass = 'bx-task';
                            $iconColor = 'text-success';
                            $bgColor = 'rgba(var(--bs-success-rgb), 0.1)';
                            break;
                        case 'ticket_created':
                        case 'ticket_replied':
                            $iconClass = 'bx-support';
                            $iconColor = 'text-info';
                            $bgColor = 'rgba(var(--bs-info-rgb), 0.1)';
                            break;
                        case 'withdrawal_request':
                        case 'withdrawal_approved':
                        case 'balance_approved':
                            $iconClass = 'bx-money';
                            $iconColor = 'text-warning';
                            $bgColor = 'rgba(var(--bs-warning-rgb), 0.1)';
                            break;
                        case 'announcement':
                            $iconClass = 'bx-megaphone';
                            $iconColor = 'text-danger';
                            $bgColor = 'rgba(var(--bs-danger-rgb), 0.1)';
                            break;
                    }
                @endphp
                
                <div class="notification-card {{ $notification->status == 1 ? 'unread' : 'read' }}" 
                     data-id="{{ $notification->id }}" 
                     data-type="{{ $notification->type }}"
                     data-link="{{ $notification->link ?? '#' }}">
                    <div class="d-flex align-items-start">
                        <div class="notification-icon {{ $iconColor }}" style="background: {{ $bgColor }};">
                            <i class="bx {{ $iconClass }}"></i>
                        </div>
                        <div class="notification-content">
                            <div class="notification-title {{ $notification->status == 1 ? 'fw-bold' : '' }}">
                                {{ $notification->title }}
                                @if($notification->status == 1)
                                    <span class="badge bg-primary ms-2">{{__('New')}}</span>
                                @endif
                            </div>
                            <div class="notification-desc">
                                {{ $notification->description }}
                            </div>
                            <div class="notification-time">
                                <i class="bx bx-time-five me-1"></i>
                                {{ $notification->created_at->diffForHumans() }}
                                @if($notification->expires_at)
                                    <span class="ms-3">
                                        <i class="bx bx-calendar me-1"></i>
                                        {{__('Expires')}}: {{ $notification->expires_at->diffForHumans() }}
                                    </span>
                                @endif
                            </div>
                            <div class="notification-actions">
                                @if($notification->status == 1)
                                    <button class="btn btn-outline-primary btn-modern mark-read-btn" data-id="{{ $notification->id }}">
                                        <i class="bx bx-check me-1"></i>{{__('Mark as Read')}}
                                    </button>
                                @endif
                                @if($notification->link && $notification->link !== '#')
                                    <button class="btn btn-primary btn-modern view-btn" data-link="{{ $notification->link }}">
                                        <i class="bx bx-link-external me-1"></i>{{__('View')}}
                                    </button>
                                @endif
                                <button class="btn btn-outline-danger btn-modern delete-btn" data-id="{{ $notification->id }}">
                                    <i class="bx bx-trash me-1"></i>{{__('Delete')}}
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            @empty
                <div class="empty-state">
                    <div class="empty-icon">
                        <i class="bx bx-bell-off"></i>
                    </div>
                    <h4>{{__('No Notifications')}}</h4>
                    <p>{{__('You have no notifications at the moment. Check back later!')}}</p>
                </div>
            @endforelse
        </div>

        <!-- Load More Button -->
        @if($notifications->hasPages())
            <div class="text-center mt-4">
                <a href="{{ $notifications->nextPageUrl() }}" class="btn btn-outline-primary btn-lg">
                    <i class="bx bx-loader-alt me-2"></i>{{__('Load More')}}
                </a>
            </div>
        @endif
    </div>
</div>
@endsection

@section('script')
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Filter functionality
    document.querySelectorAll('.filter-tab').forEach(tab => {
        tab.addEventListener('click', function() {
            // Update active tab
            document.querySelectorAll('.filter-tab').forEach(t => t.classList.remove('active'));
            this.classList.add('active');
            
            const filter = this.dataset.filter;
            const notifications = document.querySelectorAll('.notification-card');
            
            notifications.forEach(notification => {
                if (filter === 'all') {
                    notification.style.display = 'block';
                } else if (filter === 'unread') {
                    notification.style.display = notification.classList.contains('unread') ? 'block' : 'none';
                } else {
                    notification.style.display = notification.dataset.type === filter ? 'block' : 'none';
                }
            });
        });
    });
    
    // Mark as read functionality
    document.querySelectorAll('.mark-read-btn').forEach(btn => {
        btn.addEventListener('click', function(e) {
            e.stopPropagation();
            const notificationId = this.dataset.id;
            markAsRead(notificationId, this);
        });
    });
    
    // View functionality
    document.querySelectorAll('.view-btn').forEach(btn => {
        btn.addEventListener('click', function(e) {
            e.stopPropagation();
            const link = this.dataset.link;
            if (link && link !== '#') {
                window.location.href = link;
            }
        });
    });
    
    // Delete functionality
    document.querySelectorAll('.delete-btn').forEach(btn => {
        btn.addEventListener('click', function(e) {
            e.stopPropagation();
            const notificationId = this.dataset.id;
            deleteNotification(notificationId, this);
        });
    });
});

function markAsRead(notificationId, button) {
    fetch(`/mark-notification-read/${notificationId}`, {
        method: 'POST',
        headers: {
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
            'Content-Type': 'application/json',
        }
    }).then(response => {
        if (response.ok) {
            const card = button.closest('.notification-card');
            card.classList.remove('unread');
            card.classList.add('read');
            
            // Remove "New" badge
            const badge = card.querySelector('.badge');
            if (badge) badge.remove();
            
            // Remove bold from title
            const title = card.querySelector('.notification-title');
            if (title) title.classList.remove('fw-bold');
            
            // Remove button
            button.remove();
            
            // Update stats
            updateStats();
            
            showToast('{{__('Notification marked as read')}}', 'success');
        }
    }).catch(error => {
        console.error('Error:', error);
        showToast('{{__('Error marking notification as read')}}', 'error');
    });
}

function deleteNotification(notificationId, button) {
    if (confirm('{{__('Are you sure you want to delete this notification?')}}')) {
        fetch(`/notifications/${notificationId}`, {
            method: 'DELETE',
            headers: {
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
                'Content-Type': 'application/json',
            }
        }).then(response => {
            if (response.ok) {
                const card = button.closest('.notification-card');
                card.style.opacity = '0';
                card.style.transform = 'translateX(-100%)';
                
                setTimeout(() => {
                    card.remove();
                    updateStats();
                    checkEmptyState();
                }, 300);
                
                showToast('{{__('Notification deleted')}}', 'success');
            }
        }).catch(error => {
            console.error('Error:', error);
            showToast('{{__('Error deleting notification')}}', 'error');
        });
    }
}

function markAllAsRead() {
    if (confirm('{{__('Mark all notifications as read?')}}')) {
        fetch('/mark-all-notifications-read', {
            method: 'POST',
            headers: {
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
                'Content-Type': 'application/json',
            }
        }).then(response => {
            if (response.ok) {
                // Update all unread cards
                document.querySelectorAll('.notification-card.unread').forEach(card => {
                    card.classList.remove('unread');
                    card.classList.add('read');
                    
                    const badge = card.querySelector('.badge');
                    if (badge) badge.remove();
                    
                    const title = card.querySelector('.notification-title');
                    if (title) title.classList.remove('fw-bold');
                    
                    const button = card.querySelector('.mark-read-btn');
                    if (button) button.remove();
                });
                
                updateStats();
                showToast('{{__('All notifications marked as read')}}', 'success');
            }
        }).catch(error => {
            console.error('Error:', error);
            showToast('{{__('Error marking all notifications as read')}}', 'error');
        });
    }
}

function updateStats() {
    const total = document.querySelectorAll('.notification-card').length;
    const unread = document.querySelectorAll('.notification-card.unread').length;
    const read = total - unread;
    
    document.getElementById('totalCount').textContent = total;
    document.getElementById('unreadCountStat').textContent = unread;
    document.getElementById('readCount').textContent = read;
}

function checkEmptyState() {
    const notifications = document.querySelectorAll('.notification-card');
    if (notifications.length === 0) {
        document.getElementById('notificationsContainer').innerHTML = `
            <div class="empty-state">
                <div class="empty-icon">
                    <i class="bx bx-bell-off"></i>
                </div>
                <h4>{{__('No Notifications')}}</h4>
                <p>{{__('You have no notifications at the moment. Check back later!')}}</p>
            </div>
        `;
    }
}

function showToast(message, type) {
    // Simple toast notification
    const toast = document.createElement('div');
    toast.className = `alert alert-${type === 'success' ? 'success' : 'danger'} position-fixed`;
    toast.style.cssText = 'top: 20px; right: 20px; z-index: 9999; min-width: 300px;';
    toast.textContent = message;
    
    document.body.appendChild(toast);
    
    setTimeout(() => {
        toast.remove();
    }, 3000);
}
</script>
@endsection