@extends('layouts.app')

@section('style')
<link href="{{asset('assets/plugins/datatable/css/dataTables.bootstrap5.min.css')}}" rel="stylesheet">
<link href="{{asset('assets/plugins/datatable/css/responsive.bootstrap5.min.css')}}" rel="stylesheet">
@endsection

@section('wrapper')
<div class="page-wrapper">
    <div class="page-content">
        <!-- Header -->
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-3">
            <div class="breadcrumb-title pe-3">{{__('Admin')}}</div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ url('/home') }}"><i class="bx bx-home-alt"></i></a></li>
                        <li class="breadcrumb-item active" aria-current="page">{{__('Orders Management')}}</li>
                    </ol>
                </nav>
            </div>
        </div>

        <!-- Statistics Cards -->
        <div class="row mb-4">
            <div class="col-lg-2 col-md-4 col-sm-6">
                <div class="card stats-card">
                    <div class="card-body text-center">
                        <div class="stats-icon bg-primary">
                            <i class="bx bx-shopping-bag"></i>
                        </div>
                        <div class="stats-number">{{ $stats['total'] }}</div>
                        <div class="stats-title">{{__('Total Orders')}}</div>
                    </div>
                </div>
            </div>
            <div class="col-lg-2 col-md-4 col-sm-6">
                <div class="card stats-card">
                    <div class="card-body text-center">
                        <div class="stats-icon bg-warning">
                            <i class="bx bx-time"></i>
                        </div>
                        <div class="stats-number">{{ $stats['pending'] }}</div>
                        <div class="stats-title">{{__('Pending')}}</div>
                    </div>
                </div>
            </div>
            <div class="col-lg-2 col-md-4 col-sm-6">
                <div class="card stats-card">
                    <div class="card-body text-center">
                        <div class="stats-icon bg-info">
                            <i class="bx bx-loader"></i>
                        </div>
                        <div class="stats-number">{{ $stats['processing'] }}</div>
                        <div class="stats-title">{{__('Processing')}}</div>
                    </div>
                </div>
            </div>
            <div class="col-lg-2 col-md-4 col-sm-6">
                <div class="card stats-card">
                    <div class="card-body text-center">
                        <div class="stats-icon bg-success">
                            <i class="bx bx-check-circle"></i>
                        </div>
                        <div class="stats-number">{{ $stats['completed'] }}</div>
                        <div class="stats-title">{{__('Completed')}}</div>
                    </div>
                </div>
            </div>
            <div class="col-lg-2 col-md-4 col-sm-6">
                <div class="card stats-card">
                    <div class="card-body text-center">
                        <div class="stats-icon bg-success">
                            <i class="bx bx-money"></i>
                        </div>
                        <div class="stats-number">{{ number_format($stats['revenue_today'], 2) }}₺</div>
                        <div class="stats-title">{{__('Today Revenue')}}</div>
                    </div>
                </div>
            </div>
            <div class="col-lg-2 col-md-4 col-sm-6">
                <div class="card stats-card">
                    <div class="card-body text-center">
                        <div class="stats-icon bg-primary">
                            <i class="bx bx-trending-up"></i>
                        </div>
                        <div class="stats-number">{{ number_format($stats['revenue_month'], 2) }}₺</div>
                        <div class="stats-title">{{__('Month Revenue')}}</div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Main Card -->
        <div class="card">
            <div class="card-header">
                <div class="d-flex justify-content-between align-items-center">
                    <h5 class="mb-0">{{__('Orders Management')}}</h5>
                    <div class="d-flex gap-2">
                        <button class="btn btn-primary btn-sm" onclick="refreshData()">
                            <i class="bx bx-refresh me-2"></i>{{__('Refresh')}}
                        </button>
                    </div>
                </div>
            </div>
            <div class="card-body">
                <!-- Filters -->
                <div class="row mb-3">
                    <div class="col-md-3">
                        <select class="form-select form-select-sm" id="statusFilter">
                            <option value="">{{__('All Status')}}</option>
                            <option value="pending">{{__('Pending')}}</option>
                            <option value="processing">{{__('Processing')}}</option>
                            <option value="completed">{{__('Completed')}}</option>
                            <option value="cancelled">{{__('Cancelled')}}</option>
                        </select>
                    </div>
                    <div class="col-md-3">
                        <select class="form-select form-select-sm" id="paymentStatusFilter">
                            <option value="">{{__('All Payment Status')}}</option>
                            <option value="pending">{{__('Payment Pending')}}</option>
                            <option value="paid">{{__('Paid')}}</option>
                            <option value="failed">{{__('Failed')}}</option>
                            <option value="refunded">{{__('Refunded')}}</option>
                        </select>
                    </div>
                </div>

                <!-- Desktop Table -->
                <div class="d-none d-lg-block">
                    <div class="table-responsive">
                        <table id="ordersTable" class="table table-striped">
                            <thead>
                                <tr>
                                    <th>{{__('Order')}}</th>
                                    <th>{{__('Customer')}}</th>
                                    <th>{{__('Product')}}</th>
                                    <th>{{__('Amount')}}</th>
                                    <th>{{__('Status')}}</th>
                                    <th>{{__('Payment')}}</th>
                                    <th>{{__('Date')}}</th>
                                    <th>{{__('Actions')}}</th>
                                </tr>
                            </thead>
                            <tbody>
                                <!-- DataTable will populate this -->
                            </tbody>
                        </table>
                    </div>
                </div>

                <!-- Mobile Cards -->
                <div class="d-lg-none">
                    <div class="mobile-header mb-3">
                        <div class="d-flex justify-content-between align-items-center">
                            <h6 class="mb-0">{{__('Orders')}}</h6>
                            <button class="btn btn-primary btn-sm" onclick="refreshMobileData()">
                                <i class="bx bx-refresh"></i>
                            </button>
                        </div>
                    </div>

                    <!-- Mobile Filters -->
                    <div class="row mb-3">
                        <div class="col-6">
                            <select class="form-select form-select-sm" id="statusFilterMobile">
                                <option value="">{{__('All Status')}}</option>
                                <option value="pending">{{__('Pending')}}</option>
                                <option value="processing">{{__('Processing')}}</option>
                                <option value="completed">{{__('Completed')}}</option>
                                <option value="cancelled">{{__('Cancelled')}}</option>
                            </select>
                        </div>
                        <div class="col-6">
                            <select class="form-select form-select-sm" id="paymentStatusFilterMobile">
                                <option value="">{{__('All Payment')}}</option>
                                <option value="pending">{{__('Pending')}}</option>
                                <option value="paid">{{__('Paid')}}</option>
                                <option value="failed">{{__('Failed')}}</option>
                                <option value="refunded">{{__('Refunded')}}</option>
                            </select>
                        </div>
                    </div>

                    <!-- Mobile Container -->
                    <div id="mobileOrdersContainer">
                        <!-- Mobile cards will be loaded here -->
                    </div>

                    <div class="text-center mt-3">
                        <button class="btn btn-outline-primary" id="loadMoreBtn" onclick="loadMoreMobileData()">
                            <i class="bx bx-loader-alt bx-spin d-none" id="loadMoreSpinner"></i>
                            {{__('Load More')}}
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Order Status Update Offcanvas -->
<div class="offcanvas offcanvas-end" tabindex="-1" id="updateOrderOffcanvas" aria-labelledby="updateOrderOffcanvasLabel">
    <div class="offcanvas-header">
        <h5 class="offcanvas-title" id="updateOrderOffcanvasLabel">{{__('Update Order Status')}}</h5>
        <button type="button" class="btn-close" data-bs-dismiss="offcanvas" aria-label="Close"></button>
    </div>
    <div class="offcanvas-body">
        <form id="updateOrderForm">
            @csrf
            <input type="hidden" id="updateOrderId" name="order_id">
            
            <!-- Order Info -->
            <div class="mb-3">
                <div class="card bg-light">
                    <div class="card-body">
                        <h6 id="orderInfoTitle">{{__('Order Information')}}</h6>
                        <div id="orderInfoContent">
                            <!-- Order details will be loaded here -->
                        </div>
                    </div>
                </div>
            </div>

            <!-- Status -->
            <div class="mb-3">
                <label for="updateStatus" class="form-label">{{__('Order Status')}} <span class="text-danger">*</span></label>
                <select class="form-select" id="updateStatus" name="status" required>
                    <option value="pending">{{__('Pending')}}</option>
                    <option value="processing">{{__('Processing')}}</option>
                    <option value="completed">{{__('Completed')}}</option>
                    <option value="cancelled">{{__('Cancelled')}}</option>
                </select>
            </div>

            <!-- Notes -->
            <div class="mb-3">
                <label for="updateNotes" class="form-label">{{__('Notes')}} <span class="text-muted">({{__('Optional')}})</span></label>
                <textarea class="form-control" id="updateNotes" name="notes" rows="3" placeholder="{{__('Add notes about this status change...')}}"></textarea>
            </div>

            <!-- Submit Button -->
            <div class="d-grid">
                <button type="submit" class="btn btn-primary" id="updateOrderBtn">
                    <i class="bx bx-check me-2"></i>{{__('Update Status')}}
                    <span class="spinner-border spinner-border-sm ms-2 d-none" id="updateOrderSpinner"></span>
                </button>
            </div>
        </form>
    </div>
</div>
@endsection

@section('script')
<script src="{{asset('assets/plugins/datatable/js/jquery.dataTables.min.js')}}"></script>
<script src="{{asset('assets/plugins/datatable/js/dataTables.bootstrap5.min.js')}}"></script>
<script src="{{asset('assets/plugins/datatable/js/dataTables.responsive.min.js')}}"></script>
<script src="{{asset('assets/plugins/datatable/js/responsive.bootstrap5.min.js')}}"></script>

<script>
let ordersTable;
let mobileCurrentPage = 0;
let mobileHasMore = true;

$(document).ready(function() {
    // Initialize DataTable
    ordersTable = $('#ordersTable').DataTable({
        processing: true,
        serverSide: true,
        ajax: {
            url: '{{ route('api.admin.orders.datatable') }}',
            type: 'GET',
            data: function(d) {
                d.status = $('#statusFilter').val();
                d.payment_status = $('#paymentStatusFilter').val();
            },
            error: function(xhr, error, thrown) {
                Lobibox.notify('error', {
                    title: '{{__('Error')}}',
                    msg: '{{__('Error loading data')}}: ' + xhr.status,
                    sound: false
                });
            }
        },
        columns: [
            { 
                data: 'order_number', 
                name: 'order_number'
            },
            { 
                data: 'user_name', 
                name: 'user_name',
                render: function(data, type, row) {
                    return `
                        <div>
                            <strong>${data}</strong><br>
                            <small class="text-muted">${row.user_email}</small>
                        </div>
                    `;
                }
            },
            { 
                data: 'product_name', 
                name: 'product_name',
                render: function(data, type, row) {
                    return `
                        <div>
                            ${data}<br>
                            <small class="text-muted">{{__('Qty')}}: ${row.quantity}</small>
                        </div>
                    `;
                }
            },
            { 
                data: 'total_amount', 
                name: 'total_amount'
            },
            { 
                data: 'status_text', 
                name: 'status',
                render: function(data, type, row) {
                    return `<span class="badge bg-${row.status_badge}">${data}</span>`;
                }
            },
            { 
                data: 'payment_status_text', 
                name: 'payment_status',
                render: function(data, type, row) {
                    return `<span class="badge bg-${row.payment_status_badge}">${data}</span>`;
                }
            },
            { 
                data: 'created_at', 
                name: 'created_at'
            },
            { 
                data: 'id', 
                name: 'actions',
                orderable: false,
                searchable: false,
                width: '150px',
                render: function(data, type, row) {
                    return `
                        <div class="d-flex gap-1">
                            <button class="btn btn-sm btn-primary" onclick="viewOrder(${data})" title="{{__('View')}}">
                                <i class="bx bx-show me-1"></i>{{__('View')}}
                            </button>
                            <button class="btn btn-sm btn-warning" onclick="updateOrderStatus(${data})" title="{{__('Update')}}">
                                <i class="bx bx-edit me-1"></i>{{__('Update')}}
                            </button>
                        </div>
                    `;
                }
            }
        ],
        order: [[6, 'desc']], // Order by created_at desc
        pageLength: 25,
        responsive: true,
        language: {
            processing: "{{__('Processing...')}}",
            search: "{{__('Search:')}}",
            lengthMenu: "{{__('Show _MENU_ entries')}}",
            info: "{{__('Showing _START_ to _END_ of _TOTAL_ entries')}}",
            infoEmpty: "{{__('Showing 0 to 0 of 0 entries')}}",
            infoFiltered: "{{__('(filtered from _MAX_ total entries)')}}",
            paginate: {
                first: "{{__('First')}}",
                last: "{{__('Last')}}",
                next: "{{__('Next')}}",
                previous: "{{__('Previous')}}"
            }
        }
    });

    // Filter change handlers
    $('#statusFilter, #paymentStatusFilter').on('change', function() {
        ordersTable.ajax.reload();
    });

    // Mobile initialization
    if (window.innerWidth < 992) {
        loadMobileData();
    }

    // Mobile filters
    $('#statusFilterMobile, #paymentStatusFilterMobile').on('change', function() {
        mobileCurrentPage = 0;
        mobileHasMore = true;
        loadMobileData(true);
    });

    // Initialize offcanvas events
    initializeOffcanvasEvents();
});

function loadMobileData(reset = false) {
    if (reset) {
        $('#mobileOrdersContainer').empty();
        mobileCurrentPage = 0;
    }

    $('#loadMoreSpinner').removeClass('d-none');
    $('#loadMoreBtn').prop('disabled', true);

    const status = $('#statusFilterMobile').val();
    const paymentStatus = $('#paymentStatusFilterMobile').val();

    $.ajax({
        url: '{{ route('api.admin.orders.datatable') }}',
        method: 'GET',
        data: {
            start: mobileCurrentPage * 10,
            length: 10,
            status: status,
            payment_status: paymentStatus
        },
        success: function(response) {
            if (response.data && response.data.length > 0) {
                response.data.forEach(function(order) {
                    const card = createMobileOrderCard(order);
                    $('#mobileOrdersContainer').append(card);
                });
                mobileCurrentPage++;
                mobileHasMore = response.data.length === 10;
            } else {
                mobileHasMore = false;
                if (mobileCurrentPage === 0) {
                    $('#mobileOrdersContainer').html(`
                        <div class="text-center py-5">
                            <i class="bx bx-shopping-bag display-1 text-muted"></i>
                            <h6 class="mt-3">{{__('No orders found')}}</h6>
                        </div>
                    `);
                }
            }
            
            $('#loadMoreBtn').toggle(mobileHasMore);
        },
        error: function() {
            Lobibox.notify('error', {
                title: '{{__('Error')}}',
                msg: '{{__('Error loading mobile data')}}',
                sound: false
            });
        },
        complete: function() {
            $('#loadMoreSpinner').addClass('d-none');
            $('#loadMoreBtn').prop('disabled', false);
        }
    });
}

function createMobileOrderCard(order) {
    return `
        <div class="card mb-3 order-mobile-card">
            <div class="card-body">
                <div class="d-flex justify-content-between align-items-start mb-2">
                    <h6 class="card-title mb-0">${order.order_number}</h6>
                    <span class="badge bg-${order.status_badge}">${order.status_text}</span>
                </div>
                
                <div class="order-info mb-3">
                    <div><strong>{{__('Customer')}}:</strong> ${order.user_name}</div>
                    <div><strong>{{__('Product')}}:</strong> ${order.product_name}</div>
                    <div><strong>{{__('Amount')}}:</strong> ${order.total_amount}</div>
                    <div><strong>{{__('Date')}}:</strong> ${order.created_at}</div>
                </div>
                
                <div class="d-flex justify-content-between align-items-center mb-2">
                    <span class="badge bg-${order.payment_status_badge}">${order.payment_status_text}</span>
                    <small class="text-muted">{{__('Qty')}}: ${order.quantity}</small>
                </div>
                
                <div class="d-grid gap-2">
                    <div class="d-flex gap-2">
                        <button class="btn btn-primary flex-fill" onclick="viewOrder(${order.id})">
                            <i class="bx bx-show me-1"></i>{{__('View Details')}}
                        </button>
                        <button class="btn btn-warning flex-fill" onclick="updateOrderStatus(${order.id})">
                            <i class="bx bx-edit me-1"></i>{{__('Update')}}
                        </button>
                    </div>
                </div>
            </div>
        </div>
    `;
}

function loadMoreMobileData() {
    if (mobileHasMore) {
        loadMobileData();
    }
}

function refreshData() {
    if (window.innerWidth >= 992) {
        ordersTable.ajax.reload();
    } else {
        refreshMobileData();
    }
    
    Lobibox.notify('info', {
        title: '{{__('Info')}}',
        msg: '{{__('Data refreshed successfully')}}',
        sound: false
    });
}

function refreshMobileData() {
    mobileCurrentPage = 0;
    mobileHasMore = true;
    loadMobileData(true);
}

function viewOrder(orderId) {
    window.open(`/orders/${orderId}`, '_blank');
}

function updateOrderStatus(orderId) {
    // Load order data for update
    $.ajax({
        url: `/orders/${orderId}`,
        method: 'GET',
        success: function(response) {
            // Populate order info
            $('#updateOrderId').val(orderId);
            $('#orderInfoContent').html(`
                <div><strong>{{__('Order')}}:</strong> ${response.order.order_number || 'N/A'}</div>
                <div><strong>{{__('Customer')}}:</strong> ${response.order.user_name || 'N/A'}</div>
                <div><strong>{{__('Product')}}:</strong> ${response.order.product_name || 'N/A'}</div>
                <div><strong>{{__('Amount')}}:</strong> ${response.order.formatted_total || response.order.total_amount || 'N/A'}</div>
            `);
            $('#updateStatus').val(response.order.status);
            $('#updateNotes').val(response.order.notes || '');
            
            $('#updateOrderOffcanvas').offcanvas('show');
        },
        error: function() {
            Lobibox.notify('error', {
                title: '{{__('Error')}}',
                msg: '{{__('Error loading order data')}}',
                sound: false
            });
        }
    });
}

function initializeOffcanvasEvents() {
    // Form submission
    $('#updateOrderForm').on('submit', function(e) {
        e.preventDefault();
        
        const formData = new FormData(this);
        const orderId = $('#updateOrderId').val();
        
        $('#updateOrderBtn').prop('disabled', true);
        $('#updateOrderSpinner').removeClass('d-none');
        
        $.ajax({
            url: `/admin/orders/${orderId}/status`,
            method: 'POST',
            data: formData,
            processData: false,
            contentType: false,
            success: function(response) {
                if (response.success) {
                    Lobibox.notify('success', {
                        title: '{{__('Success')}}',
                        msg: response.message,
                        sound: false
                    });
                    
                    $('#updateOrderOffcanvas').offcanvas('hide');
                    refreshData();
                } else {
                    Lobibox.notify('error', {
                        title: '{{__('Error')}}',
                        msg: response.message,
                        sound: false
                    });
                }
            },
            error: function(xhr) {
                if (xhr.status === 422) {
                    const errors = xhr.responseJSON?.errors;
                    let errorMsg = '{{__('Please check your input and try again.')}}';
                    if (errors) {
                        const firstError = Object.values(errors)[0];
                        if (firstError && firstError[0]) {
                            errorMsg = firstError[0];
                        }
                    }
                    Lobibox.notify('error', {
                        title: '{{__('Validation Error')}}',
                        msg: errorMsg,
                        sound: false
                    });
                } else {
                    Lobibox.notify('error', {
                        title: '{{__('Error')}}',
                        msg: '{{__('Error updating order')}}',
                        sound: false
                    });
                }
            },
            complete: function() {
                $('#updateOrderBtn').prop('disabled', false);
                $('#updateOrderSpinner').addClass('d-none');
            }
        });
    });
}
</script>
@endsection

<style>
.stats-card {
    transition: transform 0.2s ease;
}

.stats-card:hover {
    transform: translateY(-2px);
}

.stats-icon {
    width: 50px;
    height: 50px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    margin: 0 auto 10px;
    color: white;
    font-size: 1.5rem;
}

.stats-number {
    font-size: 1.5rem;
    font-weight: 700;
    color: #2c3e50;
    margin-bottom: 5px;
}

.stats-title {
    font-size: 0.8rem;
    color: #6c757d;
    font-weight: 500;
}

.order-mobile-card {
    transition: transform 0.2s ease;
}

.order-mobile-card:hover {
    transform: translateY(-2px);
}

.order-info div {
    margin-bottom: 0.25rem;
    font-size: 0.9rem;
}

@media (max-width: 768px) {
    .stats-card {
        margin-bottom: 1rem;
    }
    
    .stats-number {
        font-size: 1.2rem;
    }
    
    .stats-title {
        font-size: 0.75rem;
    }
}
</style>