@extends('layouts.app')

@section('wrapper')
<div class="page-wrapper">
    <div class="page-content">
        <!-- Header -->
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-3">
            <div class="breadcrumb-title pe-3">{{__('My Orders')}}</div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ url('/home') }}"><i class="bx bx-home-alt"></i></a></li>
                        <li class="breadcrumb-item active" aria-current="page">{{__('Orders')}}</li>
                    </ol>
                </nav>
            </div>
        </div>

        <!-- Filters -->
        <div class="card mb-3">
            <div class="card-body">
                <div class="row">
                    <div class="col-md-4">
                        <select class="form-select" id="statusFilter">
                            <option value="">{{__('All Orders')}}</option>
                            <option value="pending">{{__('Pending')}}</option>
                            <option value="processing">{{__('Processing')}}</option>
                            <option value="completed">{{__('Completed')}}</option>
                            <option value="cancelled">{{__('Cancelled')}}</option>
                        </select>
                    </div>
                    <div class="col-md-4">
                        <button class="btn btn-primary" onclick="refreshOrders()">
                            <i class="bx bx-refresh me-2"></i>{{__('Refresh')}}
                        </button>
                    </div>
                </div>
            </div>
        </div>

        <!-- Orders Container -->
        <div id="ordersContainer">
            <!-- Orders will be loaded here -->
        </div>

        <!-- Load More Button -->
        <div class="text-center mt-3" id="loadMoreContainer" style="display: none;">
            <button class="btn btn-outline-primary" id="loadMoreBtn" onclick="loadMoreOrders()">
                <i class="bx bx-loader-alt bx-spin d-none" id="loadMoreSpinner"></i>
                {{__('Load More')}}
            </button>
        </div>

        <!-- Empty State -->
        <div id="emptyState" class="text-center py-5" style="display: none;">
            <i class="bx bx-package display-1 text-muted"></i>
            <h5 class="mt-3">{{__('No orders found')}}</h5>
            <p class="text-muted">{{__('You haven\'t placed any orders yet.')}}</p>
            <a href="{{ route('products.index') }}" class="btn btn-primary">
                <i class="bx bx-shopping-bag me-2"></i>{{__('Browse Products')}}
            </a>
        </div>
    </div>
</div>


<!-- Order Details Offcanvas -->
<div class="offcanvas offcanvas-end" tabindex="-1" id="orderDetailsOffcanvas" aria-labelledby="orderDetailsOffcanvasLabel">
    <div class="offcanvas-header">
        <h5 class="offcanvas-title" id="orderDetailsOffcanvasLabel">{{__('Order Details')}}</h5>
        <button type="button" class="btn-close" data-bs-dismiss="offcanvas" aria-label="Close"></button>
    </div>
    <div class="offcanvas-body">
        <div id="orderDetailsOffcanvasContent">
            <!-- Order details will be loaded here -->
            <div class="text-center">
                <div class="spinner-border" role="status">
                    <span class="visually-hidden">{{__('Loading...')}}</span>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@section('script')
<script>
let currentPage = 1;
let hasMoreOrders = true;
let isLoading = false;

$(document).ready(function() {
    loadOrders();
    
    // Status filter change
    $('#statusFilter').on('change', function() {
        refreshOrders();
    });
});

function loadOrders(reset = false) {
    if (isLoading) return;
    
    if (reset) {
        currentPage = 1;
        hasMoreOrders = true;
        $('#ordersContainer').empty();
    }
    
    isLoading = true;
    $('#loadMoreSpinner').removeClass('d-none');
    $('#loadMoreBtn').prop('disabled', true);
    
    const status = $('#statusFilter').val();
    
    $.ajax({
        url: '{{ route('api.orders') }}',
        method: 'GET',
        data: {
            page: currentPage,
            status: status
        },
        success: function(response) {
            if (response.success) {
                if (response.orders.length > 0) {
                    response.orders.forEach(function(order) {
                        const orderCard = createOrderCard(order);
                        $('#ordersContainer').append(orderCard);
                    });
                    
                    hasMoreOrders = response.has_more;
                    currentPage = response.current_page + 1;
                    
                    $('#loadMoreContainer').toggle(hasMoreOrders);
                    $('#emptyState').hide();
                } else if (currentPage === 1) {
                    $('#emptyState').show();
                    $('#loadMoreContainer').hide();
                }
            }
        },
        error: function() {
            Lobibox.notify('error', {
                title: '{{__('Error')}}',
                msg: '{{__('Error loading orders')}}',
                sound: false
            });
        },
        complete: function() {
            isLoading = false;
            $('#loadMoreSpinner').addClass('d-none');
            $('#loadMoreBtn').prop('disabled', false);
        }
    });
}

function createOrderCard(order) {
    const statusBadge = getStatusBadge(order.status);
    const paymentBadge = getPaymentStatusBadge(order.payment_status);
    
    return `
        <div class="card mb-3 order-card">
            <div class="card-body">
                <div class="row">
                    <div class="col-md-8">
                        <div class="d-flex justify-content-between align-items-start mb-2">
                            <h6 class="mb-0">${order.product_name}</h6>
                            <span class="badge bg-${statusBadge.color}">${statusBadge.text}</span>
                        </div>
                        
                        <div class="order-info">
                            <small class="text-muted">
                                <strong>{{__('Order Number')}}:</strong> ${order.order_number}<br>
                                <strong>{{__('Quantity')}}:</strong> ${order.quantity}<br>
                                <strong>{{__('Order Date')}}:</strong> ${order.created_at}
                            </small>
                        </div>
                        
                        <div class="mt-2">
                            <span class="badge bg-${paymentBadge.color} me-2">${paymentBadge.text}</span>
                            ${order.delivered_at ? `<span class="badge bg-success">{{__('Delivered')}}: ${order.delivered_at}</span>` : ''}
                        </div>
                    </div>
                    
                    <div class="col-md-4 text-end">
                        <div class="order-amount mb-2">
                            <h5 class="text-primary mb-0">${order.formatted_total}</h5>
                        </div>
                        
                        <div class="order-actions">
                            <button class="btn btn-sm btn-outline-primary me-1" onclick="viewOrderDetails(${order.id})">
                                <i class="bx bx-show me-1"></i>{{__('Details')}}
                            </button>
                            
                            ${order.status === 'pending' || order.status === 'processing' ? 
                                `<button class="btn btn-sm btn-outline-danger" onclick="cancelOrder(${order.id})">
                                    <i class="bx bx-x me-1"></i>{{__('Cancel')}}
                                </button>` : ''
                            }
                        </div>
                    </div>
                </div>
            </div>
        </div>
    `;
}

function getStatusBadge(status) {
    const badges = {
        'pending': { color: 'warning', text: '{{__('Pending')}}' },
        'processing': { color: 'info', text: '{{__('Processing')}}' },
        'completed': { color: 'success', text: '{{__('Completed')}}' },
        'cancelled': { color: 'secondary', text: '{{__('Cancelled')}}' },
        'refunded': { color: 'danger', text: '{{__('Refunded')}}' }
    };
    
    return badges[status] || { color: 'secondary', text: status };
}

function getPaymentStatusBadge(paymentStatus) {
    const badges = {
        'pending': { color: 'warning', text: '{{__('Payment Pending')}}' },
        'paid': { color: 'success', text: '{{__('Paid')}}' },
        'failed': { color: 'danger', text: '{{__('Payment Failed')}}' },
        'refunded': { color: 'secondary', text: '{{__('Refunded')}}' }
    };
    
    return badges[paymentStatus] || { color: 'secondary', text: paymentStatus };
}

function refreshOrders() {
    loadOrders(true);
    
    Lobibox.notify('info', {
        title: '{{__('Info')}}',
        msg: '{{__('Orders refreshed')}}',
        sound: false
    });
}

function loadMoreOrders() {
    if (hasMoreOrders && !isLoading) {
        loadOrders();
    }
}

function viewOrderDetails(orderId) {
    // Show offcanvas with loading
    const offcanvas = new bootstrap.Offcanvas(document.getElementById('orderDetailsOffcanvas'));
    offcanvas.show();
    
    // Load order details
    $.ajax({
        url: `/api/orders/${orderId}`,
        method: 'GET',
        success: function(response) {
            if (response.success) {
                const order = response.order;
                const orderDetailsHtml = generateOrderDetailsHtml(order);
                $('#orderDetailsOffcanvasContent').html(orderDetailsHtml);
            }
        },
        error: function() {
            $('#orderDetailsOffcanvasContent').html(`
                <div class="alert alert-danger">
                    <i class="bx bx-error me-2"></i>
                    {{__('Error loading order details')}}
                </div>
            `);
            
            Lobibox.notify('error', {
                title: '{{__('Error')}}',
                msg: '{{__('Error loading order details')}}',
                sound: false
            });
        }
    });
}

function generateOrderDetailsHtml(order) {
    const digitalContent = order.product_data?.digital_content || '';
    const canCancel = order.can_be_cancelled;
    
    return `
        <div class="order-details">
            <!-- Order Header -->
            <div class="card mb-3">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-start mb-3">
                        <div>
                            <h6 class="mb-1">${order.order_number}</h6>
                            <small class="text-muted">{{__('Placed on')}} ${order.created_at}</small>
                        </div>
                        <div class="text-end">
                            <span class="badge bg-${order.status_badge} mb-1">${order.status_text}</span><br>
                            <span class="badge bg-${order.payment_status_badge}">${order.payment_status_text}</span>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-6">
                            <small class="text-muted">{{__('Total Amount')}}</small>
                            <div class="fw-bold">${order.formatted_total}</div>
                        </div>
                        <div class="col-6">
                            <small class="text-muted">{{__('Quantity')}}</small>
                            <div class="fw-bold">${order.quantity}</div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Product Info -->
            <div class="card mb-3">
                <div class="card-body">
                    <h6 class="card-title">{{__('Product Information')}}</h6>
                    <div class="d-flex align-items-start">
                        ${order.product?.image_url ? `
                            <img src="${order.product.image_url}" alt="${order.product_name}" 
                                 class="rounded me-3" style="width: 60px; height: 60px; object-fit: cover;">
                        ` : `
                            <div class="bg-light rounded me-3 d-flex align-items-center justify-content-center" 
                                 style="width: 60px; height: 60px;">
                                <i class="bx bx-package text-muted"></i>
                            </div>
                        `}
                        <div class="flex-grow-1">
                            <h6 class="mb-1">${order.product_name}</h6>
                            <div class="text-muted small">${order.product?.description || ''}</div>
                            <div class="mt-2">
                                <span class="badge bg-${order.product?.is_digital ? 'info' : 'secondary'}">
                                    ${order.product?.is_digital ? '{{__('Digital')}}' : '{{__('Physical')}}'}
                                </span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Digital Content (if available) -->
            ${digitalContent && order.status === 'completed' ? `
                <div class="card mb-3">
                    <div class="card-body">
                        <h6 class="card-title">{{__('Digital Content')}}</h6>
                        <div class="bg-light p-3 rounded">
                            <pre class="mb-0 small">${digitalContent}</pre>
                        </div>
                        <small class="text-muted">{{__('This content was delivered automatically')}}</small>
                    </div>
                </div>
            ` : ''}

            <!-- Delivery Info -->
            ${order.delivered_at ? `
                <div class="card mb-3">
                    <div class="card-body">
                        <h6 class="card-title">{{__('Delivery Information')}}</h6>
                        <div class="d-flex align-items-center">
                            <i class="bx bx-check-circle text-success me-2"></i>
                            <div>
                                <div class="fw-bold">{{__('Delivered')}}</div>
                                <small class="text-muted">${order.delivered_at}</small>
                            </div>
                        </div>
                    </div>
                </div>
            ` : ''}

            <!-- Notes (if any) -->
            ${order.notes ? `
                <div class="card mb-3">
                    <div class="card-body">
                        <h6 class="card-title">{{__('Notes')}}</h6>
                        <p class="mb-0">${order.notes}</p>
                    </div>
                </div>
            ` : ''}

            <!-- Actions -->
            <div class="d-grid gap-2">
                ${canCancel ? `
                    <button class="btn btn-danger" onclick="cancelOrder(${order.id})">
                        <i class="bx bx-x me-2"></i>{{__('Cancel Order')}}
                    </button>
                ` : ''}
                
                ${order.product ? `
                    <a href="/products/${order.product.id}" class="btn btn-outline-primary">
                        <i class="bx bx-package me-2"></i>{{__('View Product')}}
                    </a>
                ` : ''}
            </div>
        </div>
    `;
}

function cancelOrder(orderId) {
    Swal.fire({
        title: '{{__('Cancel Order')}}',
        text: '{{__('Are you sure you want to cancel this order?')}}',
        icon: 'warning',
        showCancelButton: true,
        confirmButtonText: '{{__('Yes, cancel it!')}}',
        cancelButtonText: '{{__('No, keep it')}}',
        confirmButtonColor: '#dc3545'
    }).then((result) => {
        if (result.isConfirmed) {
            $.ajax({
                url: `/orders/${orderId}/cancel`,
                method: 'POST',
                headers: {
                    'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
                },
                success: function(response) {
                    if (response.success) {
                        Lobibox.notify('success', {
                            title: '{{__('Success')}}',
                            msg: response.message,
                            sound: false
                        });
                        
                        refreshOrders();
                    } else {
                        Lobibox.notify('error', {
                            title: '{{__('Error')}}',
                            msg: response.message,
                            sound: false
                        });
                    }
                },
                error: function() {
                    Lobibox.notify('error', {
                        title: '{{__('Error')}}',
                        msg: '{{__('Error cancelling order')}}',
                        sound: false
                    });
                }
            });
        }
    });
}
</script>
@endsection

<style>
.order-card {
    transition: transform 0.2s ease, box-shadow 0.2s ease;
}

.order-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 8px rgba(0,0,0,0.1);
}

.order-amount h5 {
    font-weight: 700;
}

.order-info {
    line-height: 1.6;
}

.order-actions .btn {
    min-width: 80px;
}

@media (max-width: 768px) {
    .order-actions {
        margin-top: 1rem;
    }
    
    .order-actions .btn {
        width: 100%;
        margin-bottom: 0.5rem;
    }
}
</style>