@extends('layouts.app')

@section('wrapper')
<div class="page-wrapper">
    <div class="page-content">
        <!-- Header -->
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-3">
            <div class="breadcrumb-title pe-3">{{__('Order Details')}}</div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ url('/home') }}"><i class="bx bx-home-alt"></i></a></li>
                        <li class="breadcrumb-item"><a href="{{ route('orders.index') }}">{{__('Orders')}}</a></li>
                        <li class="breadcrumb-item active" aria-current="page">{{ $order->order_number }}</li>
                    </ol>
                </nav>
            </div>
        </div>

        <div class="row">
            <!-- Order Information -->
            <div class="col-lg-8">
                <!-- Order Header -->
                <div class="card mb-3">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-start mb-3">
                            <div>
                                <h4 class="mb-1">{{__('Order')}} {{ $order->order_number }}</h4>
                                <div class="text-muted">
                                    {{__('Placed on')}} {{ $order->created_at->format('d.m.Y H:i') }}
                                </div>
                            </div>
                            <div class="text-end">
                                <span class="badge bg-{{ $order->status_badge }} mb-1 fs-6">{{ $order->status_text }}</span><br>
                                <span class="badge bg-{{ $order->payment_status_badge }} fs-6">{{ $order->payment_status_text }}</span>
                            </div>
                        </div>

                        @if($order->delivered_at)
                        <div class="alert alert-success">
                            <i class="bx bx-check-circle me-2"></i>
                            {{__('Delivered on')}} {{ $order->delivered_at->format('d.m.Y H:i') }}
                        </div>
                        @endif

                        @if($order->notes)
                        <div class="alert alert-info">
                            <strong>{{__('Notes')}}:</strong> {{ $order->notes }}
                        </div>
                        @endif
                    </div>
                </div>

                <!-- Product Information -->
                <div class="card mb-3">
                    <div class="card-header">
                        <h5 class="mb-0">{{__('Product Information')}}</h5>
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <div class="col-md-3">
                                @if($order->product && $order->product->image_url)
                                <img src="{{ $order->product->image_url }}" class="img-fluid rounded" alt="{{ $order->product_name }}">
                                @else
                                <div class="placeholder-image d-flex align-items-center justify-content-center bg-light rounded">
                                    <i class="bx bx-package display-4 text-muted"></i>
                                </div>
                                @endif
                            </div>
                            <div class="col-md-9">
                                <h5>{{ $order->product_name }}</h5>
                                
                                <div class="product-details mt-3">
                                    <div class="row">
                                        <div class="col-sm-6">
                                            <strong>{{__('Unit Price')}}:</strong>
                                            <span>{{ number_format($order->product_price, 2) }}₺</span>
                                        </div>
                                        <div class="col-sm-6">
                                            <strong>{{__('Quantity')}}:</strong>
                                            <span>{{ $order->quantity }}</span>
                                        </div>
                                        <div class="col-sm-6">
                                            <strong>{{__('Total Amount')}}:</strong>
                                            <span class="text-primary fw-bold">{{ $order->formatted_total }}</span>
                                        </div>
                                        @if($order->product)
                                        <div class="col-sm-6">
                                            <strong>{{__('Product Type')}}:</strong>
                                            <span class="badge bg-{{ $order->product->is_digital ? 'info' : 'warning' }}">
                                                {{ $order->product->is_digital ? __('Digital') : __('Physical') }}
                                            </span>
                                        </div>
                                        @endif
                                    </div>
                                </div>

                                @if($order->product)
                                <div class="mt-3">
                                    <a href="{{ route('products.show', $order->product->id) }}" class="btn btn-outline-primary btn-sm">
                                        <i class="bx bx-show me-2"></i>{{__('View Product')}}
                                    </a>
                                </div>
                                @endif
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Digital Content (if applicable) -->
                @if($order->delivery_data && $order->status === 'completed')
                <div class="card mb-3">
                    <div class="card-header bg-success text-white">
                        <h5 class="mb-0">
                            <i class="bx bx-download me-2"></i>{{__('Your Digital Content')}}
                        </h5>
                    </div>
                    <div class="card-body">
                        @if(is_array($order->delivery_data) && isset($order->delivery_data['content']))
                        <div class="digital-content">
                            @if(is_array($order->delivery_data['content']))
                                @foreach($order->delivery_data['content'] as $key => $value)
                                <div class="content-item mb-3 p-3 bg-light rounded">
                                    <strong>{{ ucfirst($key) }}:</strong>
                                    <div class="mt-2">
                                        <code class="user-select-all">{{ $value }}</code>
                                        <button class="btn btn-sm btn-outline-secondary ms-2" onclick="copyToClipboard('{{ $value }}')">
                                            <i class="bx bx-copy"></i>
                                        </button>
                                    </div>
                                </div>
                                @endforeach
                            @else
                                <div class="content-item p-3 bg-light rounded">
                                    <code class="user-select-all">{{ $order->delivery_data['content'] }}</code>
                                    <button class="btn btn-sm btn-outline-secondary ms-2" onclick="copyToClipboard('{{ $order->delivery_data['content'] }}')">
                                        <i class="bx bx-copy"></i>
                                    </button>
                                </div>
                            @endif
                        </div>
                        
                        @if(isset($order->delivery_data['instructions']))
                        <div class="alert alert-info mt-3">
                            <strong>{{__('Instructions')}}:</strong>
                            <div class="mt-2">{{ $order->delivery_data['instructions'] }}</div>
                        </div>
                        @endif
                        @else
                        <div class="alert alert-info">
                            <i class="bx bx-info-circle me-2"></i>
                            {{__('Digital content will be available once the order is completed.')}}
                        </div>
                        @endif
                    </div>
                </div>
                @endif

                <!-- Order Actions -->
                @if($order->canBeCancelled())
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">{{__('Order Actions')}}</h5>
                    </div>
                    <div class="card-body">
                        <button class="btn btn-danger" onclick="cancelOrder({{ $order->id }})">
                            <i class="bx bx-x me-2"></i>{{__('Cancel Order')}}
                        </button>
                        <small class="text-muted d-block mt-2">
                            {{__('You can cancel this order and get a full refund to your account balance.')}}
                        </small>
                    </div>
                </div>
                @endif
            </div>

            <!-- Order Summary Sidebar -->
            <div class="col-lg-4">
                <!-- Order Summary -->
                <div class="card">
                    <div class="card-header">
                        <h6 class="mb-0">{{__('Order Summary')}}</h6>
                    </div>
                    <div class="card-body">
                        <div class="order-summary">
                            <div class="d-flex justify-content-between mb-2">
                                <span>{{__('Order Number')}}:</span>
                                <span class="fw-bold">{{ $order->order_number }}</span>
                            </div>
                            <div class="d-flex justify-content-between mb-2">
                                <span>{{__('Order Date')}}:</span>
                                <span>{{ $order->created_at->format('d.m.Y') }}</span>
                            </div>
                            <div class="d-flex justify-content-between mb-2">
                                <span>{{__('Order Time')}}:</span>
                                <span>{{ $order->created_at->format('H:i') }}</span>
                            </div>
                            @if($order->delivered_at)
                            <div class="d-flex justify-content-between mb-2">
                                <span>{{__('Delivered')}}:</span>
                                <span>{{ $order->delivered_at->format('d.m.Y H:i') }}</span>
                            </div>
                            @endif
                            <hr>
                            <div class="d-flex justify-content-between mb-2">
                                <span>{{__('Subtotal')}}:</span>
                                <span>{{ $order->formatted_total }}</span>
                            </div>
                            <div class="d-flex justify-content-between mb-2">
                                <span>{{__('Payment Method')}}:</span>
                                <span>{{__('Account Balance')}}</span>
                            </div>
                            <hr>
                            <div class="d-flex justify-content-between">
                                <span class="fw-bold">{{__('Total')}}:</span>
                                <span class="fw-bold text-primary">{{ $order->formatted_total }}</span>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Customer Information -->
                @if(auth()->user()->user_type == 1)
                <div class="card mt-3">
                    <div class="card-header">
                        <h6 class="mb-0">{{__('Customer Information')}}</h6>
                    </div>
                    <div class="card-body">
                        <div class="customer-info">
                            <div class="mb-2">
                                <strong>{{__('Name')}}:</strong>
                                <span>{{ $order->user->account_holder ?? $order->user->name }}</span>
                            </div>
                            <div class="mb-2">
                                <strong>{{__('Email')}}:</strong>
                                <span>{{ $order->user->email }}</span>
                            </div>
                            @if($order->user->phone)
                            <div class="mb-2">
                                <strong>{{__('Phone')}}:</strong>
                                <span>{{ $order->user->phone }}</span>
                            </div>
                            @endif
                        </div>
                    </div>
                </div>
                @endif

                <!-- Support -->
                <div class="card mt-3">
                    <div class="card-header">
                        <h6 class="mb-0">{{__('Need Help?')}}</h6>
                    </div>
                    <div class="card-body">
                        <p class="mb-3">{{__('If you have any questions about your order, please contact our support team.')}}</p>
                        <a href="{{ route('tickets.index') }}" class="btn btn-outline-primary w-100">
                            <i class="bx bx-support me-2"></i>{{__('Contact Support')}}
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@section('script')
<script>
function copyToClipboard(text) {
    navigator.clipboard.writeText(text).then(function() {
        Lobibox.notify('success', {
            title: '{{__('Copied')}}',
            msg: '{{__('Content copied to clipboard')}}',
            sound: false
        });
    }).catch(function() {
        // Fallback for older browsers
        const textArea = document.createElement('textarea');
        textArea.value = text;
        document.body.appendChild(textArea);
        textArea.select();
        document.execCommand('copy');
        document.body.removeChild(textArea);
        
        Lobibox.notify('success', {
            title: '{{__('Copied')}}',
            msg: '{{__('Content copied to clipboard')}}',
            sound: false
        });
    });
}

function cancelOrder(orderId) {
    Swal.fire({
        title: '{{__('Cancel Order')}}',
        text: '{{__('Are you sure you want to cancel this order? The amount will be refunded to your account balance.')}}',
        icon: 'warning',
        showCancelButton: true,
        confirmButtonText: '{{__('Yes, cancel it!')}}',
        cancelButtonText: '{{__('No, keep it')}}',
        confirmButtonColor: '#dc3545'
    }).then((result) => {
        if (result.isConfirmed) {
            $.ajax({
                url: `/orders/${orderId}/cancel`,
                method: 'POST',
                headers: {
                    'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
                },
                success: function(response) {
                    if (response.success) {
                        Swal.fire({
                            title: '{{__('Order Cancelled')}}',
                            text: response.message,
                            icon: 'success',
                            confirmButtonText: '{{__('OK')}}',
                            confirmButtonColor: '#28a745'
                        }).then(() => {
                            location.reload();
                        });
                    } else {
                        Lobibox.notify('error', {
                            title: '{{__('Error')}}',
                            msg: response.message,
                            sound: false
                        });
                    }
                },
                error: function() {
                    Lobibox.notify('error', {
                        title: '{{__('Error')}}',
                        msg: '{{__('Error cancelling order')}}',
                        sound: false
                    });
                }
            });
        }
    });
}
</script>
@endsection

<style>
.placeholder-image {
    height: 150px;
    background-color: #f8f9fa;
}

.order-summary {
    font-size: 0.9rem;
}

.digital-content .content-item {
    border: 1px solid #dee2e6;
}

.digital-content code {
    background: #f8f9fa;
    padding: 0.5rem;
    border-radius: 0.25rem;
    display: block;
    word-break: break-all;
    font-size: 0.9rem;
}

.customer-info {
    font-size: 0.9rem;
}

.badge.fs-6 {
    font-size: 0.9rem !important;
}

@media (max-width: 768px) {
    .placeholder-image {
        height: 120px;
        margin-bottom: 1rem;
    }
    
    .product-details .row > div {
        margin-bottom: 0.5rem;
    }
    
    .digital-content code {
        font-size: 0.8rem;
        padding: 0.25rem;
    }
}
</style>