@extends('layouts.app')

@section('style')
<link href="{{asset('assets/plugins/datatable/css/dataTables.bootstrap5.min.css')}}" rel="stylesheet">
<link href="{{asset('assets/plugins/datatable/css/responsive.bootstrap5.min.css')}}" rel="stylesheet">
@endsection

@section('wrapper')
<div class="page-wrapper">
    <div class="page-content">
        <!-- Header -->
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-3">
            <div class="breadcrumb-title pe-3">{{__('Admin')}}</div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ url('/home') }}"><i class="bx bx-home-alt"></i></a></li>
                        <li class="breadcrumb-item active" aria-current="page">{{__('Products Management')}}</li>
                    </ol>
                </nav>
            </div>
        </div>

        <!-- Statistics Cards -->
        <div class="row mb-4">
            <div class="col-lg-2 col-md-4 col-sm-6">
                <div class="card stats-card">
                    <div class="card-body text-center">
                        <div class="stats-icon bg-primary">
                            <i class="bx bx-package"></i>
                        </div>
                        <div class="stats-number">{{ $stats['total'] }}</div>
                        <div class="stats-title">{{__('Total Products')}}</div>
                    </div>
                </div>
            </div>
            <div class="col-lg-2 col-md-4 col-sm-6">
                <div class="card stats-card">
                    <div class="card-body text-center">
                        <div class="stats-icon bg-success">
                            <i class="bx bx-check-circle"></i>
                        </div>
                        <div class="stats-number">{{ $stats['active'] }}</div>
                        <div class="stats-title">{{__('Active')}}</div>
                    </div>
                </div>
            </div>
            <div class="col-lg-2 col-md-4 col-sm-6">
                <div class="card stats-card">
                    <div class="card-body text-center">
                        <div class="stats-icon bg-info">
                            <i class="bx bx-cloud"></i>
                        </div>
                        <div class="stats-number">{{ $stats['digital'] }}</div>
                        <div class="stats-title">{{__('Digital')}}</div>
                    </div>
                </div>
            </div>
            <div class="col-lg-2 col-md-4 col-sm-6">
                <div class="card stats-card">
                    <div class="card-body text-center">
                        <div class="stats-icon bg-warning">
                            <i class="bx bx-error"></i>
                        </div>
                        <div class="stats-number">{{ $stats['low_stock'] }}</div>
                        <div class="stats-title">{{__('Low Stock')}}</div>
                    </div>
                </div>
            </div>
            <div class="col-lg-2 col-md-4 col-sm-6">
                <div class="card stats-card">
                    <div class="card-body text-center">
                        <div class="stats-icon bg-danger">
                            <i class="bx bx-x-circle"></i>
                        </div>
                        <div class="stats-number">{{ $stats['out_of_stock'] }}</div>
                        <div class="stats-title">{{__('Out of Stock')}}</div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Main Card -->
        <div class="card">
            <div class="card-header">
                <div class="d-flex justify-content-between align-items-center">
                    <h5 class="mb-0">{{__('Products Management')}}</h5>
                    <div class="d-flex gap-2">
                        <button class="btn btn-success btn-sm" data-bs-toggle="offcanvas" data-bs-target="#createProductOffcanvas">
                            <i class="bx bx-plus me-2"></i>{{__('Add Product')}}
                        </button>
                        <button class="btn btn-primary btn-sm" onclick="refreshData()">
                            <i class="bx bx-refresh me-2"></i>{{__('Refresh')}}
                        </button>
                    </div>
                </div>
            </div>
            <div class="card-body">
                <!-- Filters -->
                <div class="row mb-3">
                    <div class="col-md-3">
                        <select class="form-select form-select-sm" id="categoryFilter">
                            <option value="">{{__('All Categories')}}</option>
                            <!-- Categories will be loaded via Ajax -->
                        </select>
                    </div>
                    <div class="col-md-3">
                        <select class="form-select form-select-sm" id="typeFilter">
                            <option value="">{{__('All Types')}}</option>
                            <option value="digital">{{__('Digital')}}</option>
                            <option value="physical">{{__('Physical')}}</option>
                        </select>
                    </div>
                    <div class="col-md-3">
                        <select class="form-select form-select-sm" id="statusFilter">
                            <option value="">{{__('All Status')}}</option>
                            <option value="1">{{__('Active')}}</option>
                            <option value="0">{{__('Inactive')}}</option>
                        </select>
                    </div>
                </div>

                <!-- Desktop Table -->
                <div class="d-none d-lg-block">
                    <div class="table-responsive">
                        <table id="productsTable" class="table table-striped">
                            <thead>
                                <tr>
                                    <th>{{__('Product')}}</th>
                                    <th>{{__('Category')}}</th>
                                    <th>{{__('Price')}}</th>
                                    <th>{{__('Type')}}</th>
                                    <th>{{__('Stock')}}</th>
                                    <th>{{__('Status')}}</th>
                                    <th>{{__('Actions')}}</th>
                                </tr>
                            </thead>
                            <tbody>
                                <!-- DataTable will populate this -->
                            </tbody>
                        </table>
                    </div>
                </div>

                <!-- Mobile Cards -->
                <div class="d-lg-none">
                    <div class="mobile-header mb-3">
                        <div class="d-flex justify-content-between align-items-center">
                            <h6 class="mb-0">{{__('Products')}}</h6>
                            <button class="btn btn-primary btn-sm" onclick="refreshMobileData()">
                                <i class="bx bx-refresh"></i>
                            </button>
                        </div>
                    </div>

                    <!-- Mobile Filters -->
                    <div class="row mb-3">
                        <div class="col-4">
                            <select class="form-select form-select-sm" id="categoryFilterMobile">
                                <option value="">{{__('All Categories')}}</option>
                            </select>
                        </div>
                        <div class="col-4">
                            <select class="form-select form-select-sm" id="typeFilterMobile">
                                <option value="">{{__('All Types')}}</option>
                                <option value="digital">{{__('Digital')}}</option>
                                <option value="physical">{{__('Physical')}}</option>
                            </select>
                        </div>
                        <div class="col-4">
                            <select class="form-select form-select-sm" id="statusFilterMobile">
                                <option value="">{{__('All Status')}}</option>
                                <option value="1">{{__('Active')}}</option>
                                <option value="0">{{__('Inactive')}}</option>
                            </select>
                        </div>
                    </div>

                    <!-- Mobile Container -->
                    <div id="mobileProductsContainer">
                        <!-- Mobile cards will be loaded here -->
                    </div>

                    <div class="text-center mt-3">
                        <button class="btn btn-outline-primary" id="loadMoreBtn" onclick="loadMoreMobileData()">
                            <i class="bx bx-loader-alt bx-spin d-none" id="loadMoreSpinner"></i>
                            {{__('Load More')}}
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Create Product Offcanvas -->
<div class="offcanvas offcanvas-end" tabindex="-1" id="createProductOffcanvas" aria-labelledby="createProductOffcanvasLabel">
    <div class="offcanvas-header">
        <h5 class="offcanvas-title" id="createProductOffcanvasLabel">{{__('Add New Product')}}</h5>
        <button type="button" class="btn-close" data-bs-dismiss="offcanvas" aria-label="Close"></button>
    </div>
    <div class="offcanvas-body">
        <form id="createProductForm">
            @csrf
            
            <!-- Product Name -->
            <div class="mb-3">
                <label for="createProductName" class="form-label">{{__('Product Name')}} <span class="text-danger">*</span></label>
                <input type="text" class="form-control" id="createProductName" name="product_name" required maxlength="255">
            </div>

            <!-- Description -->
            <div class="mb-3">
                <label for="createDescription" class="form-label">{{__('Description')}} <span class="text-danger">*</span></label>
                <textarea class="form-control" id="createDescription" name="description" rows="3" required></textarea>
            </div>

            <!-- Category -->
            <div class="mb-3">
                <label for="createCategory" class="form-label">{{__('Category')}} <span class="text-danger">*</span></label>
                <select class="form-select" id="createCategory" name="category_id" required>
                    <option value="">{{__('Select Category')}}</option>
                </select>
            </div>

            <!-- Price -->
            <div class="mb-3">
                <label for="createPrice" class="form-label">{{__('Price')}} <span class="text-danger">*</span></label>
                <div class="input-group">
                    <input type="number" class="form-control" id="createPrice" name="price" step="0.01" min="0" required>
                    <span class="input-group-text">₺</span>
                </div>
            </div>

            <!-- Product Type -->
            <div class="mb-3">
                <label for="createIsDigital" class="form-label">{{__('Product Type')}} <span class="text-danger">*</span></label>
                <select class="form-select" id="createIsDigital" name="is_digital" required>
                    <option value="1">{{__('Digital Product')}}</option>
                    <option value="0">{{__('Physical Product')}}</option>
                </select>
            </div>

            <!-- Stock Quantity (for physical products) -->
            <div class="mb-3" id="createStockDiv" style="display: none;">
                <label for="createStock" class="form-label">{{__('Stock Quantity')}}</label>
                <input type="number" class="form-control" id="createStock" name="stock_quantity" min="0" value="0">
            </div>

            <!-- Digital Content (for digital products) -->
            <div class="mb-3" id="createDigitalContentDiv">
                <label for="createDigitalContent" class="form-label">{{__('Digital Content')}}</label>
                <textarea class="form-control" id="createDigitalContent" name="digital_content" rows="3" placeholder="{{__('Enter digital content (codes, links, instructions, etc.)')}}"></textarea>
            </div>

            <!-- Auto Delivery -->
            <div class="mb-3" id="createAutoDeliveryDiv">
                <div class="form-check">
                    <input class="form-check-input" type="checkbox" id="createAutoDelivery" name="auto_delivery" value="1" checked>
                    <label class="form-check-label" for="createAutoDelivery">
                        {{__('Auto Delivery')}}
                    </label>
                </div>
            </div>

            <!-- Min/Max Quantity -->
            <div class="row">
                <div class="col-6">
                    <div class="mb-3">
                        <label for="createMinQuantity" class="form-label">{{__('Min Quantity')}}</label>
                        <input type="number" class="form-control" id="createMinQuantity" name="min_quantity" min="1" value="1">
                    </div>
                </div>
                <div class="col-6">
                    <div class="mb-3">
                        <label for="createMaxQuantity" class="form-label">{{__('Max Quantity')}}</label>
                        <input type="number" class="form-control" id="createMaxQuantity" name="max_quantity" min="1" value="10">
                    </div>
                </div>
            </div>

            <!-- Provider -->
            <div class="mb-3">
                <label for="createProvider" class="form-label">{{__('Provider')}} <span class="text-danger">*</span></label>
                <input type="text" class="form-control" id="createProvider" name="provider" required>
            </div>

            <!-- Submit Button -->
            <div class="d-grid">
                <button type="submit" class="btn btn-primary" id="createProductBtn">
                    <i class="bx bx-plus me-2"></i>{{__('Add Product')}}
                    <span class="spinner-border spinner-border-sm ms-2 d-none" id="createProductSpinner"></span>
                </button>
            </div>
        </form>
    </div>
</div>

<!-- Edit Product Offcanvas -->
<div class="offcanvas offcanvas-end" tabindex="-1" id="editProductOffcanvas" aria-labelledby="editProductOffcanvasLabel">
    <div class="offcanvas-header">
        <h5 class="offcanvas-title" id="editProductOffcanvasLabel">{{__('Edit Product')}}</h5>
        <button type="button" class="btn-close" data-bs-dismiss="offcanvas" aria-label="Close"></button>
    </div>
    <div class="offcanvas-body">
        <form id="editProductForm">
            @csrf
            @method('POST')
            <input type="hidden" id="editProductId" name="product_id">
            
            <!-- Product Name -->
            <div class="mb-3">
                <label for="editProductName" class="form-label">{{__('Product Name')}} <span class="text-danger">*</span></label>
                <input type="text" class="form-control" id="editProductName" name="product_name" required maxlength="255">
            </div>

            <!-- Description -->
            <div class="mb-3">
                <label for="editDescription" class="form-label">{{__('Description')}} <span class="text-danger">*</span></label>
                <textarea class="form-control" id="editDescription" name="description" rows="3" required></textarea>
            </div>

            <!-- Category -->
            <div class="mb-3">
                <label for="editCategory" class="form-label">{{__('Category')}} <span class="text-danger">*</span></label>
                <select class="form-select" id="editCategory" name="category_id" required>
                    <option value="">{{__('Select Category')}}</option>
                </select>
            </div>

            <!-- Price -->
            <div class="mb-3">
                <label for="editPrice" class="form-label">{{__('Price')}} <span class="text-danger">*</span></label>
                <div class="input-group">
                    <input type="number" class="form-control" id="editPrice" name="price" step="0.01" min="0" required>
                    <span class="input-group-text">₺</span>
                </div>
            </div>

            <!-- Product Type -->
            <div class="mb-3">
                <label for="editIsDigital" class="form-label">{{__('Product Type')}} <span class="text-danger">*</span></label>
                <select class="form-select" id="editIsDigital" name="is_digital" required>
                    <option value="1">{{__('Digital Product')}}</option>
                    <option value="0">{{__('Physical Product')}}</option>
                </select>
            </div>

            <!-- Stock Quantity (for physical products) -->
            <div class="mb-3" id="editStockDiv" style="display: none;">
                <label for="editStock" class="form-label">{{__('Stock Quantity')}}</label>
                <input type="number" class="form-control" id="editStock" name="stock_quantity" min="0" value="0">
            </div>

            <!-- Digital Content (for digital products) -->
            <div class="mb-3" id="editDigitalContentDiv">
                <label for="editDigitalContent" class="form-label">{{__('Digital Content')}}</label>
                <textarea class="form-control" id="editDigitalContent" name="digital_content" rows="3" placeholder="{{__('Enter digital content (codes, links, instructions, etc.)')}}"></textarea>
            </div>

            <!-- Auto Delivery -->
            <div class="mb-3" id="editAutoDeliveryDiv">
                <div class="form-check">
                    <input class="form-check-input" type="checkbox" id="editAutoDelivery" name="auto_delivery" value="1">
                    <label class="form-check-label" for="editAutoDelivery">
                        {{__('Auto Delivery')}}
                    </label>
                </div>
            </div>

            <!-- Min/Max Quantity -->
            <div class="row">
                <div class="col-6">
                    <div class="mb-3">
                        <label for="editMinQuantity" class="form-label">{{__('Min Quantity')}}</label>
                        <input type="number" class="form-control" id="editMinQuantity" name="min_quantity" min="1" value="1">
                    </div>
                </div>
                <div class="col-6">
                    <div class="mb-3">
                        <label for="editMaxQuantity" class="form-label">{{__('Max Quantity')}}</label>
                        <input type="number" class="form-control" id="editMaxQuantity" name="max_quantity" min="1" value="10">
                    </div>
                </div>
            </div>

            <!-- Provider -->
            <div class="mb-3">
                <label for="editProvider" class="form-label">{{__('Provider')}} <span class="text-danger">*</span></label>
                <input type="text" class="form-control" id="editProvider" name="provider" required>
            </div>

            <!-- Image URL (hidden for now, will be handled separately) -->
            <input type="hidden" id="editImageUrl" name="image_url" value="">

            <!-- Status -->
            <div class="mb-3">
                <label for="editStatus" class="form-label">{{__('Status')}} <span class="text-danger">*</span></label>
                <select class="form-select" id="editStatus" name="status" required>
                    <option value="1">{{__('Active')}}</option>
                    <option value="0">{{__('Inactive')}}</option>
                </select>
            </div>

            <!-- Submit Button -->
            <div class="d-grid">
                <button type="submit" class="btn btn-warning" id="editProductBtn">
                    <i class="bx bx-save me-2"></i>{{__('Update Product')}}
                    <span class="spinner-border spinner-border-sm ms-2 d-none" id="editProductSpinner"></span>
                </button>
            </div>
        </form>
    </div>
</div>
@endsection

@section('script')
<script src="{{asset('assets/plugins/datatable/js/jquery.dataTables.min.js')}}"></script>
<script src="{{asset('assets/plugins/datatable/js/dataTables.bootstrap5.min.js')}}"></script>
<script src="{{asset('assets/plugins/datatable/js/dataTables.responsive.min.js')}}"></script>
<script src="{{asset('assets/plugins/datatable/js/responsive.bootstrap5.min.js')}}"></script>

<script>
let productsTable;
let mobileCurrentPage = 0;
let mobileHasMore = true;

$(document).ready(function() {
    // Initialize DataTable
    productsTable = $('#productsTable').DataTable({
        processing: true,
        serverSide: true,
        ajax: {
            url: '{{ route('api.admin.products.datatable') }}',
            type: 'GET',
            data: function(d) {
                d.category_id = $('#categoryFilter').val();
                d.type = $('#typeFilter').val();
                d.status = $('#statusFilter').val();
            },
            error: function(xhr, error, thrown) {
                Lobibox.notify('error', {
                    title: '{{__('Error')}}',
                    msg: '{{__('Error loading data')}}: ' + xhr.status,
                    sound: false
                });
            }
        },
        columns: [
            { 
                data: 'product_name', 
                name: 'product_name'
            },
            { 
                data: 'category_name', 
                name: 'category_name'
            },
            { 
                data: 'price', 
                name: 'price'
            },
            { 
                data: 'is_digital', 
                name: 'is_digital',
                render: function(data) {
                    return data ? '<span class="badge bg-info">{{__('Digital')}}</span>' : '<span class="badge bg-secondary">{{__('Physical')}}</span>';
                }
            },
            { 
                data: 'stock_status', 
                name: 'stock_status',
                render: function(data, type, row) {
                    return `<span class="badge bg-${row.stock_badge}">${data}</span>`;
                }
            },
            { 
                data: 'status_text', 
                name: 'status',
                render: function(data, type, row) {
                    return `<span class="badge bg-${row.status_badge}">${data}</span>`;
                }
            },
            { 
                data: 'id', 
                name: 'actions',
                orderable: false,
                searchable: false,
                width: '150px',
                render: function(data, type, row) {
                    return `
                        <div class="d-flex gap-1">
                            <button class="btn btn-sm btn-warning" onclick="editProduct(${data})" title="{{__('Edit')}}">
                                <i class="bx bx-edit me-1"></i>{{__('Edit')}}
                            </button>
                            <button class="btn btn-sm btn-danger" onclick="deleteProduct(${data})" title="{{__('Delete')}}">
                                <i class="bx bx-trash me-1"></i>{{__('Delete')}}
                            </button>
                        </div>
                    `;
                }
            }
        ],
        order: [[0, 'asc']],
        pageLength: 25,
        responsive: true,
        language: {
            processing: "{{__('Processing...')}}",
            search: "{{__('Search:')}}",
            lengthMenu: "{{__('Show _MENU_ entries')}}",
            info: "{{__('Showing _START_ to _END_ of _TOTAL_ entries')}}",
            infoEmpty: "{{__('Showing 0 to 0 of 0 entries')}}",
            infoFiltered: "{{__('(filtered from _MAX_ total entries)')}}",
            paginate: {
                first: "{{__('First')}}",
                last: "{{__('Last')}}",
                next: "{{__('Next')}}",
                previous: "{{__('Previous')}}"
            }
        }
    });

    // Filter change handlers
    $('#categoryFilter, #typeFilter, #statusFilter').on('change', function() {
        productsTable.ajax.reload();
    });

    // Mobile initialization
    if (window.innerWidth < 992) {
        loadMobileData();
    }

    // Mobile filters
    $('#categoryFilterMobile, #typeFilterMobile, #statusFilterMobile').on('change', function() {
        mobileCurrentPage = 0;
        mobileHasMore = true;
        loadMobileData(true);
    });

    // Load categories
    loadCategories();

    // Initialize offcanvas events
    initializeOffcanvasEvents();
});

function loadCategories() {
    // Load categories for filters and form
    $.ajax({
        url: '/api/product-categories',
        method: 'GET',
        success: function(response) {
            if (response.success) {
                const options = response.categories.map(cat => 
                    `<option value="${cat.id}">${cat.name}</option>`
                ).join('');
                
                $('#categoryFilter, #categoryFilterMobile, #createCategory').append(options);
            }
        }
    });
}

function loadCategoriesForEdit(selectedCategoryId = null) {
    // Edit form için kategorileri yükle
    $.ajax({
        url: '/api/product-categories',
        method: 'GET',
        success: function(response) {
            if (response.success) {
                const options = response.categories.map(cat => 
                    `<option value="${cat.id}">${cat.name}</option>`
                ).join('');
                
                $('#editCategory').html('<option value="">{{__('Select Category')}}</option>' + options);
                
                // Seçili kategoriyi ayarla
                if (selectedCategoryId) {
                    $('#editCategory').val(selectedCategoryId);
                }
            }
        }
    });
}

function refreshData() {
    if (window.innerWidth >= 992) {
        productsTable.ajax.reload();
    } else {
        refreshMobileData();
    }
    
    Lobibox.notify('info', {
        title: '{{__('Info')}}',
        msg: '{{__('Data refreshed successfully')}}',
        sound: false
    });
}

function refreshMobileData() {
    mobileCurrentPage = 0;
    mobileHasMore = true;
    loadMobileData(true);
}

function loadMobileData(reset = false) {
    // Mobile data loading implementation
    console.log('Mobile data loading...');
}

function loadMoreMobileData() {
    if (mobileHasMore) {
        loadMobileData();
    }
}

function initializeOffcanvasEvents() {
    // Product type change handler for create form
    $('#createIsDigital').on('change', function() {
        if ($(this).val() === '0') {
            $('#createStockDiv').show();
            $('#createDigitalContentDiv, #createAutoDeliveryDiv').hide();
        } else {
            $('#createStockDiv').hide();
            $('#createDigitalContentDiv, #createAutoDeliveryDiv').show();
        }
    });

    // Product type change handler for edit form
    $('#editIsDigital').on('change', function() {
        if ($(this).val() === '0') {
            $('#editStockDiv').show();
            $('#editDigitalContentDiv, #editAutoDeliveryDiv').hide();
        } else {
            $('#editStockDiv').hide();
            $('#editDigitalContentDiv, #editAutoDeliveryDiv').show();
        }
    });

    // Create form submission
    $('#createProductForm').on('submit', function(e) {
        e.preventDefault();
        console.log('Product form submitted');
        // Implementation will be added
    });

    // Edit form submission
    $('#editProductForm').on('submit', function(e) {
        e.preventDefault();
        
        // Form validation
        let isValid = true;
        let errorMessages = [];
        
        // Required fields validation
        const productName = $('#editProductName').val().trim();
        const description = $('#editDescription').val().trim();
        const price = $('#editPrice').val();
        const categoryId = $('#editCategory').val();
        const minQuantity = $('#editMinQuantity').val();
        const maxQuantity = $('#editMaxQuantity').val();
        const provider = $('#editProvider').val().trim();
        
        if (!productName) {
            isValid = false;
            errorMessages.push('{{__('Product name is required')}}');
            $('#editProductName').addClass('is-invalid');
        } else {
            $('#editProductName').removeClass('is-invalid');
        }
        
        if (!description) {
            isValid = false;
            errorMessages.push('{{__('Description is required')}}');
            $('#editDescription').addClass('is-invalid');
        } else {
            $('#editDescription').removeClass('is-invalid');
        }
        
        if (!price || price <= 0) {
            isValid = false;
            errorMessages.push('{{__('Price must be greater than 0')}}');
            $('#editPrice').addClass('is-invalid');
        } else {
            $('#editPrice').removeClass('is-invalid');
        }
        
        if (!categoryId) {
            isValid = false;
            errorMessages.push('{{__('Category is required')}}');
            $('#editCategory').addClass('is-invalid');
        } else {
            $('#editCategory').removeClass('is-invalid');
        }
        
        if (!provider) {
            isValid = false;
            errorMessages.push('{{__('Provider is required')}}');
            $('#editProvider').addClass('is-invalid');
        } else {
            $('#editProvider').removeClass('is-invalid');
        }
        
        if (!minQuantity || minQuantity < 1) {
            isValid = false;
            errorMessages.push('{{__('Min quantity must be at least 1')}}');
            $('#editMinQuantity').addClass('is-invalid');
        } else {
            $('#editMinQuantity').removeClass('is-invalid');
        }
        
        if (!maxQuantity || maxQuantity < 1) {
            isValid = false;
            errorMessages.push('{{__('Max quantity must be at least 1')}}');
            $('#editMaxQuantity').addClass('is-invalid');
        } else {
            $('#editMaxQuantity').removeClass('is-invalid');
        }
        
        if (parseInt(minQuantity) > parseInt(maxQuantity)) {
            isValid = false;
            errorMessages.push('{{__('Min quantity cannot be greater than max quantity')}}');
            $('#editMinQuantity, #editMaxQuantity').addClass('is-invalid');
        }
        
        // Fiziksel ürün için stok kontrolü
        const isDigital = $('#editIsDigital').val();
        if (isDigital === '0') {
            const stockQuantity = $('#editStock').val();
            if (!stockQuantity || stockQuantity < 0) {
                isValid = false;
                errorMessages.push('{{__('Stock quantity is required for physical products')}}');
                $('#editStock').addClass('is-invalid');
            } else {
                $('#editStock').removeClass('is-invalid');
            }
        }
        
        if (!isValid) {
            Lobibox.notify('error', {
                title: '{{__('Validation Error')}}',
                msg: errorMessages.join('<br>'),
                sound: false
            });
            return;
        }
        
        const productId = $('#editProductId').val();
        const formData = new FormData(this);
        
        $('#editProductBtn').prop('disabled', true);
        $('#editProductSpinner').removeClass('d-none');
        
        $.ajax({
            url: `/products/${productId}`,
            method: 'POST',
            data: formData,
            processData: false,
            contentType: false,
            headers: {
                'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            },
            success: function(response) {
                if (response.success) {
                    Lobibox.notify('success', {
                        title: '{{__('Success')}}',
                        msg: response.message || '{{__('Product updated successfully')}}',
                        sound: false
                    });
                    
                    // Offcanvas'ı kapat
                    const offcanvas = bootstrap.Offcanvas.getInstance(document.getElementById('editProductOffcanvas'));
                    offcanvas.hide();
                    
                    // Tabloyu yenile
                    refreshData();
                }
            },
            error: function(xhr) {
                let errorMsg = '{{__('Error updating product')}}';
                if (xhr.responseJSON && xhr.responseJSON.message) {
                    errorMsg = xhr.responseJSON.message;
                }
                
                Lobibox.notify('error', {
                    title: '{{__('Error')}}',
                    msg: errorMsg,
                    sound: false
                });
            },
            complete: function() {
                $('#editProductBtn').prop('disabled', false);
                $('#editProductSpinner').addClass('d-none');
            }
        });
    });
}

function editProduct(id) {
    // Ürün bilgilerini getir ve edit formunu doldur
    $.ajax({
        url: `/products/${id}`,
        method: 'GET',
        headers: {
            'Accept': 'application/json'
        },
        success: function(response) {
            if (response.success) {
                const product = response.product;
                
                // Form alanlarını doldur
                $('#editProductId').val(product.id);
                $('#editProductName').val(product.product_name);
                $('#editDescription').val(product.description);
                $('#editPrice').val(product.price);
                $('#editIsDigital').val(product.is_digital.toString());
                $('#editStock').val(product.stock_quantity || 0);
                $('#editDigitalContent').val(product.digital_content || '');
                $('#editAutoDelivery').prop('checked', product.auto_delivery == 1);
                $('#editMinQuantity').val(product.min_quantity || 1);
                $('#editMaxQuantity').val(product.max_quantity || 10);
                $('#editProvider').val(product.provider || '');
                $('#editStatus').val(product.status.toString());
                $('#editImageUrl').val(product.image_url || '');
                
                // Ürün tipine göre alanları göster/gizle
                if (product.is_digital == 0) {
                    $('#editStockDiv').show();
                    $('#editDigitalContentDiv, #editAutoDeliveryDiv').hide();
                } else {
                    $('#editStockDiv').hide();
                    $('#editDigitalContentDiv, #editAutoDeliveryDiv').show();
                }
                
                // Kategorileri yükle ve seç
                loadCategoriesForEdit(product.category_id);
                
                // Offcanvas'ı aç
                const offcanvas = new bootstrap.Offcanvas(document.getElementById('editProductOffcanvas'));
                offcanvas.show();
            }
        },
        error: function(xhr) {
            Lobibox.notify('error', {
                title: '{{__('Error')}}',
                msg: '{{__('Error loading product data')}}',
                sound: false
            });
        }
    });
}

function deleteProduct(id) {
    Swal.fire({
        title: '{{__('Are you sure?')}}',
        text: '{{__('This product will be deleted permanently!')}}',
        icon: 'warning',
        showCancelButton: true,
        confirmButtonText: '{{__('Yes, delete it!')}}',
        cancelButtonText: '{{__('Cancel')}}',
        confirmButtonColor: '#dc3545'
    }).then((result) => {
        if (result.isConfirmed) {
            $.ajax({
                url: `/products/${id}/delete`,
                method: 'POST',
                headers: {
                    'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
                },
                success: function(response) {
                    if (response.success) {
                        Swal.fire({
                            title: '{{__('Deleted!')}}',
                            text: response.message || '{{__('Product has been deleted successfully.')}}',
                            icon: 'success',
                            timer: 2000,
                            showConfirmButton: false
                        });
                        
                        // Tabloyu yenile
                        refreshData();
                    }
                },
                error: function(xhr) {
                    let errorMsg = '{{__('Error deleting product')}}';
                    if (xhr.responseJSON && xhr.responseJSON.message) {
                        errorMsg = xhr.responseJSON.message;
                    }
                    
                    Swal.fire({
                        title: '{{__('Error!')}}',
                        text: errorMsg,
                        icon: 'error'
                    });
                }
            });
        }
    });
}
</script>
@endsection

<style>
.stats-card {
    transition: transform 0.2s ease;
}

.stats-card:hover {
    transform: translateY(-2px);
}

.stats-icon {
    width: 50px;
    height: 50px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    margin: 0 auto 10px;
    color: white;
    font-size: 1.5rem;
}

.stats-number {
    font-size: 2rem;
    font-weight: 700;
    color: #2c3e50;
    margin-bottom: 5px;
}

.stats-title {
    font-size: 0.9rem;
    color: #6c757d;
    font-weight: 500;
}

@media (max-width: 768px) {
    .stats-card {
        margin-bottom: 1rem;
    }
}
</style>