@extends('layouts.app')

@section('wrapper')
<div class="page-wrapper">
    <div class="page-content">
        <!-- Header -->
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-3">
            <div class="breadcrumb-title pe-3">{{__('Products')}}</div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ url('/home') }}"><i class="bx bx-home-alt"></i></a></li>
                        <li class="breadcrumb-item"><a href="{{ route('products.index') }}">{{__('Products')}}</a></li>
                        <li class="breadcrumb-item active" aria-current="page">{{ $product->product_name }}</li>
                    </ol>
                </nav>
            </div>
        </div>

        <div class="row">
            <!-- Product Details -->
            <div class="col-lg-8">
                <div class="card">
                    <div class="card-body">
                        <div class="row">
                            <!-- Product Image -->
                            <div class="col-md-6">
                                <div class="product-image-container">
                                    <img src="{{ $product->image_url ?: '/assets/images/products/placeholder.png' }}" 
                                         class="img-fluid rounded product-main-image" 
                                         alt="{{ $product->product_name }}"
                                         id="productMainImage">
                                </div>
                            </div>
                            
                            <!-- Product Info -->
                            <div class="col-md-6">
                                <div class="product-info">
                                    <h3 class="product-title">{{ $product->product_name }}</h3>
                                    
                                    <!-- Price -->
                                    <div class="product-price mb-3">
                                        <h2 class="text-primary mb-0">{{ $product->formatted_price }}</h2>
                                    </div>
                                    
                                    <!-- Category & Type -->
                                    <div class="product-meta mb-3">
                                        @if($product->category)
                                        <span class="badge bg-secondary me-2">{{ $product->category->name }}</span>
                                        @endif
                                        
                                        @if($product->is_digital)
                                        <span class="badge bg-info me-2">{{__('Digital Product')}}</span>
                                        @else
                                        <span class="badge bg-warning me-2">{{__('Physical Product')}}</span>
                                        @endif
                                        
                                        <span class="badge bg-{{ $product->stock_badge }}">{{ $product->stock_status }}</span>
                                    </div>
                                    
                                    <!-- Stock Info -->
                                    @if(!$product->is_digital)
                                    <div class="stock-info mb-3">
                                        <div class="d-flex justify-content-between">
                                            <span>{{__('Available Stock')}}:</span>
                                            <span class="fw-bold">{{ $product->stock_quantity }}</span>
                                        </div>
                                    </div>
                                    @endif
                                    
                                    <!-- Quantity & Purchase -->
                                    @if($product->isAvailable())
                                    <div class="purchase-section">
                                        <form id="purchaseForm">
                                            @csrf
                                            <input type="hidden" name="product_id" value="{{ $product->id }}">
                                            
                                            <!-- Quantity -->
                                            <div class="mb-3">
                                                <label for="quantity" class="form-label">{{__('Quantity')}}</label>
                                                <div class="input-group quantity-input">
                                                    <button type="button" class="btn btn-outline-secondary" onclick="decreaseQuantity()">
                                                        <i class="bx bx-minus"></i>
                                                    </button>
                                                    <input type="number" 
                                                           class="form-control text-center" 
                                                           id="quantity" 
                                                           name="quantity" 
                                                           value="{{ $product->min_quantity }}" 
                                                           min="{{ $product->min_quantity }}" 
                                                           max="{{ $product->is_digital ? $product->max_quantity : min($product->max_quantity, $product->stock_quantity) }}"
                                                           readonly>
                                                    <button type="button" class="btn btn-outline-secondary" onclick="increaseQuantity()">
                                                        <i class="bx bx-plus"></i>
                                                    </button>
                                                </div>
                                                <small class="text-muted">
                                                    {{__('Min')}}: {{ $product->min_quantity }}, 
                                                    {{__('Max')}}: {{ $product->is_digital ? $product->max_quantity : min($product->max_quantity, $product->stock_quantity) }}
                                                </small>
                                            </div>
                                            
                                            <!-- Total Price -->
                                            <div class="total-price mb-3">
                                                <div class="d-flex justify-content-between align-items-center">
                                                    <span class="h5 mb-0">{{__('Total')}}:</span>
                                                    <span class="h4 text-primary mb-0" id="totalPrice">{{ $product->formatted_price }}</span>
                                                </div>
                                            </div>
                                            
                                            <!-- Balance Check -->
                                            @auth
                                            <div class="balance-info mb-3">
                                                <div class="d-flex justify-content-between">
                                                    <span>{{__('Your Balance')}}:</span>
                                                    <span class="fw-bold" id="userBalance">{{ number_format(auth()->user()->balance, 2) }}₺</span>
                                                </div>
                                                <div class="d-flex justify-content-between">
                                                    <span>{{__('After Purchase')}}:</span>
                                                    <span class="fw-bold" id="remainingBalance">{{ number_format(auth()->user()->balance - $product->price * $product->min_quantity, 2) }}₺</span>
                                                </div>
                                            </div>
                                            @endauth
                                            
                                            <!-- Purchase Button -->
                                            @auth
                                            <div class="d-grid">
                                                <button type="submit" class="btn btn-primary btn-lg" id="purchaseBtn">
                                                    <i class="bx bx-shopping-bag me-2"></i>{{__('Purchase Now')}}
                                                    <span class="spinner-border spinner-border-sm ms-2 d-none" id="purchaseSpinner"></span>
                                                </button>
                                            </div>
                                            @else
                                            <div class="d-grid">
                                                <a href="{{ route('login') }}" class="btn btn-primary btn-lg">
                                                    <i class="bx bx-log-in me-2"></i>{{__('Login to Purchase')}}
                                                </a>
                                            </div>
                                            @endauth
                                        </form>
                                    </div>
                                    @else
                                    <div class="alert alert-warning">
                                        <i class="bx bx-error-circle me-2"></i>
                                        {{__('This product is currently not available for purchase.')}}
                                    </div>
                                    @endif
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Product Description -->
                <div class="card mt-4">
                    <div class="card-header">
                        <h5 class="mb-0">{{__('Product Description')}}</h5>
                    </div>
                    <div class="card-body">
                        <div class="product-description">
                            {!! nl2br(e($product->description)) !!}
                        </div>
                    </div>
                </div>
                
                <!-- Digital Content Info (for digital products) -->
                @if($product->is_digital && $product->digital_content)
                <div class="card mt-4">
                    <div class="card-header">
                        <h5 class="mb-0">{{__('What You Will Get')}}</h5>
                    </div>
                    <div class="card-body">
                        <div class="digital-content-info">
                            @if(is_array($product->digital_content))
                                <ul class="list-unstyled">
                                    @foreach($product->digital_content as $item)
                                    <li class="mb-2">
                                        <i class="bx bx-check-circle text-success me-2"></i>
                                        {{ $item }}
                                    </li>
                                    @endforeach
                                </ul>
                            @else
                                <p>{{ $product->digital_content }}</p>
                            @endif
                        </div>
                        
                        @if($product->auto_delivery)
                        <div class="alert alert-info mt-3">
                            <i class="bx bx-info-circle me-2"></i>
                            {{__('This is a digital product with automatic delivery. You will receive your purchase immediately after payment.')}}
                        </div>
                        @endif
                    </div>
                </div>
                @endif
            </div>
            
            <!-- Sidebar -->
            <div class="col-lg-4">
                <!-- Product Details Card -->
                <div class="card">
                    <div class="card-header">
                        <h6 class="mb-0">{{__('Product Details')}}</h6>
                    </div>
                    <div class="card-body">
                        <div class="product-details">
                            @if($product->provider)
                            <div class="detail-item mb-2">
                                <strong>{{__('Provider')}}:</strong>
                                <span>{{ $product->provider }}</span>
                            </div>
                            @endif
                            
                            @if($product->completing_time)
                            <div class="detail-item mb-2">
                                <strong>{{__('Delivery Time')}}:</strong>
                                <span>{{ $product->completing_time }}</span>
                            </div>
                            @endif
                            
                            <div class="detail-item mb-2">
                                <strong>{{__('Product Type')}}:</strong>
                                <span>{{ $product->is_digital ? __('Digital') : __('Physical') }}</span>
                            </div>
                            
                            @if($product->is_digital && $product->auto_delivery)
                            <div class="detail-item mb-2">
                                <strong>{{__('Delivery')}}:</strong>
                                <span class="text-success">{{__('Automatic')}}</span>
                            </div>
                            @endif
                        </div>
                    </div>
                </div>
                
                <!-- Security Info -->
                <div class="card mt-3">
                    <div class="card-header">
                        <h6 class="mb-0">{{__('Purchase Security')}}</h6>
                    </div>
                    <div class="card-body">
                        <div class="security-features">
                            <div class="feature-item mb-2">
                                <i class="bx bx-shield-check text-success me-2"></i>
                                {{__('Secure Payment')}}
                            </div>
                            <div class="feature-item mb-2">
                                <i class="bx bx-support text-primary me-2"></i>
                                {{__('24/7 Support')}}
                            </div>
                            @if($product->is_digital)
                            <div class="feature-item mb-2">
                                <i class="bx bx-time text-info me-2"></i>
                                {{__('Instant Delivery')}}
                            </div>
                            @endif
                            <div class="feature-item mb-2">
                                <i class="bx bx-money text-warning me-2"></i>
                                {{__('Balance Payment')}}
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Purchase Confirmation Modal -->
<div class="modal fade" id="purchaseConfirmModal" tabindex="-1" aria-labelledby="purchaseConfirmModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="purchaseConfirmModalLabel">{{__('Confirm Purchase')}}</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div class="purchase-summary">
                    <div class="d-flex justify-content-between mb-2">
                        <span>{{__('Product')}}:</span>
                        <span class="fw-bold">{{ $product->product_name }}</span>
                    </div>
                    <div class="d-flex justify-content-between mb-2">
                        <span>{{__('Quantity')}}:</span>
                        <span class="fw-bold" id="confirmQuantity">1</span>
                    </div>
                    <div class="d-flex justify-content-between mb-2">
                        <span>{{__('Unit Price')}}:</span>
                        <span class="fw-bold">{{ $product->formatted_price }}</span>
                    </div>
                    <hr>
                    <div class="d-flex justify-content-between mb-3">
                        <span class="h6">{{__('Total Amount')}}:</span>
                        <span class="h6 text-primary" id="confirmTotal">{{ $product->formatted_price }}</span>
                    </div>
                    
                    <div class="alert alert-info">
                        <i class="bx bx-info-circle me-2"></i>
                        {{__('The amount will be deducted from your account balance.')}}
                    </div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">{{__('Cancel')}}</button>
                <button type="button" class="btn btn-primary" id="confirmPurchaseBtn">
                    <i class="bx bx-check me-2"></i>{{__('Confirm Purchase')}}
                    <span class="spinner-border spinner-border-sm ms-2 d-none" id="confirmPurchaseSpinner"></span>
                </button>
            </div>
        </div>
    </div>
</div>
@endsection

@section('script')
<script>
const productPrice = {{ $product->price }};
const minQuantity = {{ $product->min_quantity }};
const maxQuantity = {{ $product->is_digital ? $product->max_quantity : min($product->max_quantity, $product->stock_quantity) }};
@auth
const userBalance = {{ auth()->user()->balance }};
@endauth

$(document).ready(function() {
    updateTotalPrice();
    
    // Quantity input change
    $('#quantity').on('input', function() {
        updateTotalPrice();
    });
    
    // Purchase form submission
    $('#purchaseForm').on('submit', function(e) {
        e.preventDefault();
        showPurchaseConfirmation();
    });
    
    // Confirm purchase
    $('#confirmPurchaseBtn').on('click', function() {
        processPurchase();
    });
});

function increaseQuantity() {
    const quantityInput = $('#quantity');
    const currentValue = parseInt(quantityInput.val());
    
    if (currentValue < maxQuantity) {
        quantityInput.val(currentValue + 1);
        updateTotalPrice();
    }
}

function decreaseQuantity() {
    const quantityInput = $('#quantity');
    const currentValue = parseInt(quantityInput.val());
    
    if (currentValue > minQuantity) {
        quantityInput.val(currentValue - 1);
        updateTotalPrice();
    }
}

function updateTotalPrice() {
    const quantity = parseInt($('#quantity').val());
    const total = productPrice * quantity;
    
    $('#totalPrice').text(formatPrice(total));
    $('#confirmQuantity').text(quantity);
    $('#confirmTotal').text(formatPrice(total));
    
    @auth
    // Update remaining balance
    const remaining = userBalance - total;
    $('#remainingBalance').text(formatPrice(remaining));
    
    // Check if user has enough balance
    if (remaining < 0) {
        $('#purchaseBtn').prop('disabled', true).html('<i class="bx bx-error me-2"></i>{{__('Insufficient Balance')}}');
        $('#remainingBalance').addClass('text-danger');
    } else {
        $('#purchaseBtn').prop('disabled', false).html('<i class="bx bx-shopping-bag me-2"></i>{{__('Purchase Now')}}');
        $('#remainingBalance').removeClass('text-danger');
    }
    @endauth
}

function formatPrice(amount) {
    return new Intl.NumberFormat('tr-TR', {
        style: 'currency',
        currency: 'TRY',
        minimumFractionDigits: 2
    }).format(amount).replace('₺', '') + '₺';
}

function showPurchaseConfirmation() {
    $('#purchaseConfirmModal').modal('show');
}

function processPurchase() {
    const formData = new FormData($('#purchaseForm')[0]);
    
    $('#confirmPurchaseBtn').prop('disabled', true);
    $('#confirmPurchaseSpinner').removeClass('d-none');
    
    $.ajax({
        url: '{{ route('orders.store') }}',
        method: 'POST',
        data: formData,
        processData: false,
        contentType: false,
        success: function(response) {
            if (response.success) {
                $('#purchaseConfirmModal').modal('hide');
                
                Swal.fire({
                    title: '{{__('Purchase Successful!')}}',
                    html: `
                        <div class="text-start">
                            <p><strong>{{__('Order Number')}}:</strong> ${response.order.order_number}</p>
                            <p><strong>{{__('Total Amount')}}:</strong> ${response.order.total_amount}</p>
                            <p><strong>{{__('Status')}}:</strong> ${response.order.status}</p>
                        </div>
                    `,
                    icon: 'success',
                    confirmButtonText: '{{__('View My Orders')}}',
                    showCancelButton: true,
                    cancelButtonText: '{{__('Continue Shopping')}}',
                    confirmButtonColor: '#28a745'
                }).then((result) => {
                    if (result.isConfirmed) {
                        window.location.href = '{{ route('orders.index') }}';
                    } else {
                        window.location.href = '{{ route('products.index') }}';
                    }
                });
                
                // Update user balance display
                @auth
                const newBalance = userBalance - (productPrice * parseInt($('#quantity').val()));
                $('#userBalance').text(formatPrice(newBalance));
                @endauth
            } else {
                Lobibox.notify('error', {
                    title: '{{__('Purchase Failed')}}',
                    msg: response.message,
                    sound: false
                });
            }
        },
        error: function(xhr) {
            let errorMsg = '{{__('Error processing purchase')}}';
            
            if (xhr.status === 422) {
                const errors = xhr.responseJSON?.errors;
                if (errors) {
                    const firstError = Object.values(errors)[0];
                    if (firstError && firstError[0]) {
                        errorMsg = firstError[0];
                    }
                }
            } else if (xhr.responseJSON?.message) {
                errorMsg = xhr.responseJSON.message;
            }
            
            Lobibox.notify('error', {
                title: '{{__('Purchase Failed')}}',
                msg: errorMsg,
                sound: false
            });
        },
        complete: function() {
            $('#confirmPurchaseBtn').prop('disabled', false);
            $('#confirmPurchaseSpinner').addClass('d-none');
        }
    });
}
</script>
@endsection

<style>
.product-main-image {
    max-height: 400px;
    object-fit: cover;
    width: 100%;
}

.product-title {
    color: #2c3e50;
    font-weight: 600;
    margin-bottom: 1rem;
}

.product-price h2 {
    font-weight: 700;
}

.quantity-input {
    max-width: 150px;
}

.quantity-input input {
    border-left: none;
    border-right: none;
}

.quantity-input button {
    border: 1px solid #ced4da;
}

.total-price {
    background: #f8f9fa;
    padding: 1rem;
    border-radius: 0.5rem;
    border: 2px solid #e9ecef;
}

.balance-info {
    background: #e3f2fd;
    padding: 0.75rem;
    border-radius: 0.5rem;
    border: 1px solid #bbdefb;
}

.product-description {
    line-height: 1.8;
    color: #495057;
}

.digital-content-info ul li {
    padding: 0.5rem 0;
    border-bottom: 1px solid #f1f3f4;
}

.digital-content-info ul li:last-child {
    border-bottom: none;
}

.detail-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 0.5rem 0;
    border-bottom: 1px solid #f1f3f4;
}

.detail-item:last-child {
    border-bottom: none;
}

.feature-item {
    display: flex;
    align-items: center;
    font-size: 0.9rem;
}

.purchase-summary {
    background: #f8f9fa;
    padding: 1rem;
    border-radius: 0.5rem;
    margin-bottom: 1rem;
}

@media (max-width: 768px) {
    .product-main-image {
        max-height: 250px;
        margin-bottom: 1rem;
    }
    
    .quantity-input {
        max-width: 100%;
    }
    
    .total-price {
        margin-top: 1rem;
    }
}
</style>