@extends('layouts.app')

@section('wrapper')
<div class="page-wrapper">
    <div class="page-content">
        <!-- Header -->
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-3">
            <div class="breadcrumb-title pe-3">{{__('Support')}}</div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ url('/home') }}"><i class="bx bx-home-alt"></i></a></li>
                        <li class="breadcrumb-item"><a href="{{ route('tickets.index') }}">{{__('Support Tickets')}}</a></li>
                        <li class="breadcrumb-item active" aria-current="page">{{__('Create Ticket')}}</li>
                    </ol>
                </nav>
            </div>
        </div>

        <div class="row justify-content-center">
            <div class="col-lg-8">
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">{{__('Create Support Ticket')}}</h5>
                    </div>
                    <div class="card-body">
                        <form id="createTicketForm" enctype="multipart/form-data">
                            @csrf
                            
                            <!-- Subject -->
                            <div class="mb-3">
                                <label for="subject" class="form-label">{{__('Subject')}} <span class="text-danger">*</span></label>
                                <input type="text" class="form-control" id="subject" name="subject" required maxlength="255" placeholder="{{__('Brief description of your issue')}}">
                                <div class="invalid-feedback"></div>
                            </div>

                            <!-- Category -->
                            <div class="row mb-3">
                                <div class="col-md-6">
                                    <label for="category" class="form-label">{{__('Category')}} <span class="text-danger">*</span></label>
                                    <select class="form-select" id="category" name="category" required>
                                        <option value="">{{__('Select Category')}}</option>
                                        <option value="general">{{__('General')}}</option>
                                        <option value="technical">{{__('Technical')}}</option>
                                        <option value="billing">{{__('Billing')}}</option>
                                        <option value="account">{{__('Account')}}</option>
                                        <option value="withdrawal">{{__('Withdrawal')}}</option>
                                        <option value="task">{{__('Task')}}</option>
                                    </select>
                                    <div class="invalid-feedback"></div>
                                </div>
                                <div class="col-md-6">
                                    <label for="priority" class="form-label">{{__('Priority')}} <span class="text-danger">*</span></label>
                                    <select class="form-select" id="priority" name="priority" required>
                                        <option value="">{{__('Select Priority')}}</option>
                                        <option value="low">{{__('Low')}}</option>
                                        <option value="medium" selected>{{__('Medium')}}</option>
                                        <option value="high">{{__('High')}}</option>
                                        <option value="urgent">{{__('Urgent')}}</option>
                                    </select>
                                    <div class="invalid-feedback"></div>
                                </div>
                            </div>

                            <!-- Description -->
                            <div class="mb-3">
                                <label for="description" class="form-label">{{__('Description')}} <span class="text-danger">*</span></label>
                                <textarea class="form-control" id="description" name="description" rows="6" required maxlength="5000" placeholder="{{__('Please provide detailed information about your issue...')}}"></textarea>
                                <div class="form-text">{{__('Remaining characters')}}: <span id="charCount">5000</span></div>
                                <div class="invalid-feedback"></div>
                            </div>

                            <!-- File Attachments -->
                            <div class="mb-3">
                                <label for="attachments" class="form-label">{{__('Attachments')}} <span class="text-muted">({{__('Optional')}})</span></label>
                                <input type="file" class="form-control" id="attachments" name="attachments[]" multiple accept=".jpg,.jpeg,.png,.pdf,.doc,.docx,.txt">
                                <div class="form-text">
                                    {{__('Max file size: 10MB')}} | {{__('Allowed formats: JPG, PNG, PDF, DOC, TXT')}}
                                </div>
                                <div class="invalid-feedback"></div>
                                
                                <!-- File Preview -->
                                <div id="filePreview" class="mt-2"></div>
                            </div>

                            <!-- Submit Buttons -->
                            <div class="d-flex gap-2 justify-content-end">
                                <a href="{{ route('tickets.index') }}" class="btn btn-secondary">
                                    <i class="bx bx-x me-2"></i>{{__('Cancel')}}
                                </a>
                                <button type="submit" class="btn btn-primary" id="submitBtn">
                                    <i class="bx bx-check me-2"></i>{{__('Create Ticket')}}
                                    <span class="spinner-border spinner-border-sm ms-2 d-none" id="submitSpinner"></span>
                                </button>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Help Card -->
                <div class="card mt-3">
                    <div class="card-body">
                        <h6 class="card-title">{{__('Tips for Better Support')}}</h6>
                        <ul class="mb-0">
                            <li>{{__('Choose the correct category for faster response')}}</li>
                            <li>{{__('Provide detailed description of your issue')}}</li>
                            <li>{{__('Include screenshots or relevant files if applicable')}}</li>
                            <li>{{__('Set appropriate priority level')}}</li>
                        </ul>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
.file-preview-item {
    display: flex;
    align-items: center;
    justify-content: space-between;
    padding: 8px 12px;
    background-color: #f8f9fa;
    border-radius: 6px;
    margin-bottom: 8px;
}

.file-preview-info {
    display: flex;
    align-items: center;
    gap: 8px;
}

.file-preview-icon {
    width: 24px;
    height: 24px;
    display: flex;
    align-items: center;
    justify-content: center;
    border-radius: 4px;
    font-size: 12px;
    font-weight: 600;
    color: white;
}

.file-preview-icon.image { background-color: #28a745; }
.file-preview-icon.pdf { background-color: #dc3545; }
.file-preview-icon.doc { background-color: #007bff; }
.file-preview-icon.txt { background-color: #6c757d; }

.file-preview-details {
    flex: 1;
}

.file-preview-name {
    font-size: 0.9rem;
    font-weight: 500;
    color: #2c3e50;
}

.file-preview-size {
    font-size: 0.8rem;
    color: #6c757d;
}

.file-remove-btn {
    background: none;
    border: none;
    color: #dc3545;
    cursor: pointer;
    padding: 4px;
    border-radius: 4px;
    transition: background-color 0.2s;
}

.file-remove-btn:hover {
    background-color: rgba(220, 53, 69, 0.1);
}

@media (max-width: 768px) {
    .card-body {
        padding: 1rem;
    }
    
    .row.mb-3 {
        margin-bottom: 1rem !important;
    }
    
    .col-md-6 {
        margin-bottom: 1rem;
    }
}
</style>

<script>
$(document).ready(function() {
    // Character counter
    $('#description').on('input', function() {
        const remaining = 5000 - $(this).val().length;
        $('#charCount').text(remaining);
        
        if (remaining < 100) {
            $('#charCount').addClass('text-danger');
        } else {
            $('#charCount').removeClass('text-danger');
        }
    });

    // File upload handling
    $('#attachments').on('change', function() {
        const files = this.files;
        const preview = $('#filePreview');
        preview.empty();

        Array.from(files).forEach((file, index) => {
            if (file.size > 10 * 1024 * 1024) { // 10MB limit
                showNotification('error', `{{__('File')}} "${file.name}" {{__('is too large. Maximum size is 10MB.')}}`);
                return;
            }

            const fileItem = createFilePreview(file, index);
            preview.append(fileItem);
        });
    });

    // Form submission
    $('#createTicketForm').on('submit', function(e) {
        e.preventDefault();
        
        const formData = new FormData(this);
        
        // Disable submit button
        $('#submitBtn').prop('disabled', true);
        $('#submitSpinner').removeClass('d-none');
        
        // Clear previous errors
        $('.is-invalid').removeClass('is-invalid');
        $('.invalid-feedback').text('');

        $.ajax({
            url: '{{ route('tickets.store') }}',
            method: 'POST',
            data: formData,
            processData: false,
            contentType: false,
            success: function(response) {
                if (response.success) {
                    showNotification('success', response.message);
                    setTimeout(() => {
                        window.location.href = '{{ route('tickets.index') }}';
                    }, 1500);
                } else {
                    showNotification('error', response.message);
                }
            },
            error: function(xhr) {
                if (xhr.status === 422) {
                    // Validation errors
                    const errors = xhr.responseJSON.errors;
                    Object.keys(errors).forEach(field => {
                        $(`#${field}`).addClass('is-invalid');
                        $(`#${field}`).siblings('.invalid-feedback').text(errors[field][0]);
                    });
                } else {
                    showNotification('error', '{{__('Error creating ticket')}}');
                }
            },
            complete: function() {
                $('#submitBtn').prop('disabled', false);
                $('#submitSpinner').addClass('d-none');
            }
        });
    });
});

function createFilePreview(file, index) {
    const fileSize = formatFileSize(file.size);
    const fileType = getFileType(file.name);
    const fileIcon = getFileIcon(fileType);
    
    return `
        <div class="file-preview-item" data-index="${index}">
            <div class="file-preview-info">
                <div class="file-preview-icon ${fileType}">${fileIcon}</div>
                <div class="file-preview-details">
                    <div class="file-preview-name">${file.name}</div>
                    <div class="file-preview-size">${fileSize}</div>
                </div>
            </div>
            <button type="button" class="file-remove-btn" onclick="removeFile(${index})">
                <i class="bx bx-x"></i>
            </button>
        </div>
    `;
}

function getFileType(filename) {
    const ext = filename.split('.').pop().toLowerCase();
    if (['jpg', 'jpeg', 'png'].includes(ext)) return 'image';
    if (ext === 'pdf') return 'pdf';
    if (['doc', 'docx'].includes(ext)) return 'doc';
    return 'txt';
}

function getFileIcon(type) {
    const icons = {
        'image': 'IMG',
        'pdf': 'PDF',
        'doc': 'DOC',
        'txt': 'TXT'
    };
    return icons[type] || 'FILE';
}

function formatFileSize(bytes) {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
}

function removeFile(index) {
    $(`.file-preview-item[data-index="${index}"]`).remove();
    
    // Update file input
    const fileInput = document.getElementById('attachments');
    const dt = new DataTransfer();
    const files = fileInput.files;
    
    for (let i = 0; i < files.length; i++) {
        if (i !== index) {
            dt.items.add(files[i]);
        }
    }
    
    fileInput.files = dt.files;
}

function showNotification(type, message) {
    // Implement your notification system here
    if (type === 'success') {
        alert('✓ ' + message);
    } else {
        alert('✗ ' + message);
    }
}
</script>
@endsection