@extends('layouts.app')

@section('title', __('My Website Visit Tasks'))

@section('style')
    <link href="{{ asset('assets/plugins/notifications/css/lobibox.min.css') }}" rel="stylesheet">
    <link href="{{ asset('assets/plugins/datatable/css/dataTables.bootstrap5.min.css') }}" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
@endsection

@section('wrapper')
<div class="page-wrapper">
    <div class="page-content">
        <!--breadcrumb-->
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-3">
            <div class="breadcrumb-title pe-3">{{ __('My Website Visit Tasks') }}</div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ route('dashboard') }}"><i class="bx bx-home-alt"></i></a></li>
                        <li class="breadcrumb-item"><a href="{{ route('website-visits.index') }}">{{ __('Website Visits') }}</a></li>
                        <li class="breadcrumb-item active" aria-current="page">{{ __('My Tasks') }}</li>
                    </ol>
                </nav>
            </div>
            <div class="ms-auto">
                <a href="{{ route('website-visits.index') }}" class="btn btn-secondary me-2">
                    <i class='bx bx-arrow-back me-1'></i>{{ __('Back to Tasks') }}
                </a>
                <button type="button" class="btn btn-primary" onclick="openCreateOffcanvas()">
                    <i class='bx bx-plus me-1'></i>{{ __('Add New Task') }}
                </button>
            </div>
        </div>
        <!--end breadcrumb-->

        <div class="card">
            <div class="card-body">
                <div id="tasksContainer">
                    <!-- Content will be loaded here based on device type -->
                    <div class="text-center py-5">
                        <i class="bx bx-loader-alt bx-spin fs-1"></i>
                        <p>{{ __('Loading tasks...') }}</p>
                    </div>
                </div>
                    <!-- All static content removed - AJAX will handle everything -->
            </div>
        </div>
    </div>
</div>

<!-- Create Task Offcanvas -->
<div class="offcanvas offcanvas-end" tabindex="-1" id="createTaskOffcanvas" aria-labelledby="createTaskOffcanvasLabel">
    <div class="offcanvas-header">
        <h5 class="offcanvas-title" id="createTaskOffcanvasLabel">{{ __('Add New Website Visit Task') }}</h5>
        <button type="button" class="btn-close" data-bs-dismiss="offcanvas" aria-label="Close"></button>
    </div>
    <div class="offcanvas-body">
        <form id="createTaskForm">
            @csrf
            <div class="mb-3">
                <label for="create_title" class="form-label">{{ __('Task Title') }}</label>
                <input type="text" class="form-control" id="create_title" name="title" required>
            </div>
            
            <div class="mb-3">
                <label for="create_website_url" class="form-label">{{ __('Website URL') }}</label>
                <input type="url" class="form-control" id="create_website_url" name="website_url" required>
            </div>
            
            <div class="mb-3">
                <label for="create_description" class="form-label">{{ __('Description') }}</label>
                <textarea class="form-control" id="create_description" name="description" rows="3"></textarea>
            </div>
            
            <div class="mb-3">
                <label for="create_reward_amount" class="form-label">{{ __('Reward Amount') }}</label>
                <input type="number" class="form-control" id="create_reward_amount" name="reward_amount" step="0.01" min="0" required>
            </div>
            
            <div class="mb-3">
                <label for="create_visit_duration" class="form-label">{{ __('Visit Duration (seconds)') }}</label>
                <input type="number" class="form-control" id="create_visit_duration" name="visit_duration" min="10" required>
                <small class="form-text text-muted">{{ __('Minimum 10 seconds required') }}</small>
            </div>
            
            <div class="mb-3">
                <label for="create_user_daily_limit" class="form-label">{{ __('Daily Limit per User') }}</label>
                <input type="number" class="form-control" id="create_user_daily_limit" name="user_daily_limit" min="1">
                <small class="form-text text-muted">{{ __('Leave empty for unlimited') }}</small>
            </div>
            
            <div class="d-grid gap-2">
                <button type="submit" class="btn btn-primary" id="createTaskBtn">
                    <i class='bx bx-save me-1'></i>{{ __('Create Task') }}
                </button>
                <button type="button" class="btn btn-secondary" data-bs-dismiss="offcanvas">
                    <i class='bx bx-x me-1'></i>{{ __('Cancel') }}
                </button>
            </div>
        </form>
    </div>
</div>

<!-- Edit Task Offcanvas -->
<div class="offcanvas offcanvas-end" tabindex="-1" id="editTaskOffcanvas" aria-labelledby="editTaskOffcanvasLabel">
    <div class="offcanvas-header">
        <h5 class="offcanvas-title" id="editTaskOffcanvasLabel">{{ __('Edit Website Visit Task') }}</h5>
        <button type="button" class="btn-close" data-bs-dismiss="offcanvas" aria-label="Close"></button>
    </div>
    <div class="offcanvas-body">
        <form id="editTaskForm">
            @csrf
            @method('POST')
            <input type="hidden" id="edit_task_id" name="task_id">
            
            <div class="mb-3">
                <label for="edit_title" class="form-label">{{ __('Task Title') }} <span class="text-danger">*</span></label>
                <input type="text" class="form-control" id="edit_title" name="title" required>
            </div>
            
            <div class="mb-3">
                <label for="edit_description" class="form-label">{{ __('Description') }} <span class="text-danger">*</span></label>
                <textarea class="form-control" id="edit_description" name="description" rows="3" required></textarea>
            </div>
            
            <div class="mb-3">
                <label for="edit_website_url" class="form-label">{{ __('Website URL') }} <span class="text-danger">*</span></label>
                <input type="url" class="form-control" id="edit_website_url" name="website_url" required>
            </div>
            
            <div class="mb-3">
                <label for="edit_visit_method" class="form-label">{{ __('Visit Method') }} <span class="text-danger">*</span></label>
                <select class="form-control" id="edit_visit_method" name="visit_method" required>
                    <option value="direct">{{ __('Direct Visit') }}</option>
                    <option value="google_search">{{ __('Google Search') }}</option>
                </select>
            </div>
            
            <div class="mb-3" id="edit_search_keywords_row" style="display: none;">
                <label for="edit_search_keywords" class="form-label">{{ __('Search Keywords') }}</label>
                <input type="text" class="form-control" id="edit_search_keywords" name="search_keywords">
            </div>
            
            <div class="mb-3">
                <label for="edit_required_time" class="form-label">{{ __('Required Time (seconds)') }} <span class="text-danger">*</span></label>
                <input type="number" class="form-control" id="edit_required_time" name="required_time" min="10" max="3600" required>
                <small class="form-text text-muted">{{ __('Minimum time users must spend on the website') }}</small>
            </div>
            
            <div class="mb-3">
                <label for="edit_reward_amount" class="form-label">{{ __('Reward Amount') }} <span class="text-danger">*</span></label>
                <input type="number" class="form-control" id="edit_reward_amount" name="reward_amount" step="0.01" min="0.01" required>
            </div>
            
            <div class="mb-3">
                <label for="edit_max_completions" class="form-label">{{ __('Max Completions') }} <span class="text-danger">*</span></label>
                <input type="number" class="form-control" id="edit_max_completions" name="max_completions" min="1" required>
                <small class="form-text text-muted">{{ __('Total number of times this task can be completed') }}</small>
            </div>
            
            <div class="mb-3">
                <label for="edit_user_daily_limit" class="form-label">{{ __('User Daily Limit') }} <span class="text-danger">*</span></label>
                <input type="number" class="form-control" id="edit_user_daily_limit" name="user_daily_limit" min="1" max="50" required>
                <small class="form-text text-muted">{{ __('How many times per day each user can complete this task') }}</small>
            </div>
            
            <div class="mb-3">
                <label for="edit_user_total_limit" class="form-label">{{ __('User Total Limit') }} <span class="text-danger">*</span></label>
                <input type="number" class="form-control" id="edit_user_total_limit" name="user_total_limit" min="1" required>
                <small class="form-text text-muted">{{ __('How many times total each user can complete this task') }}</small>
            </div>
            
            <div class="mb-3">
                <label for="edit_target_gender" class="form-label">{{ __('Target Gender') }} <span class="text-danger">*</span></label>
                <select class="form-control" id="edit_target_gender" name="target_gender" required>
                    <option value="0">{{ __('All Genders') }}</option>
                    <option value="1">{{ __('Male') }}</option>
                    <option value="2">{{ __('Female') }}</option>
                </select>
            </div>
            
            <div class="mb-3">
                <label for="edit_target_country_id" class="form-label">{{ __('Target Country') }}</label>
                <select class="form-control" id="edit_target_country_id" name="target_country_id">
                    <option value="0">{{ __('All Countries') }}</option>
                </select>
            </div>
            
            <div class="mb-3">
                <label for="edit_target_city_id" class="form-label">{{ __('Target City') }}</label>
                <select class="form-control" id="edit_target_city_id" name="target_city_id">
                    <option value="0">{{ __('All Cities') }}</option>
                </select>
            </div>
            
            <div class="mb-3">
                <label for="edit_expires_at" class="form-label">{{ __('Expiration Date') }}</label>
                <input type="datetime-local" class="form-control" id="edit_expires_at" name="expires_at">
                <small class="form-text text-muted">{{ __('Leave empty for no expiration') }}</small>
            </div>
            
            <div class="d-grid gap-2">
                <button type="submit" class="btn btn-warning" id="editTaskBtn">
                    <i class='bx bx-save me-1'></i>{{ __('Update Task') }}
                </button>
                <button type="button" class="btn btn-secondary" data-bs-dismiss="offcanvas">
                    <i class='bx bx-x me-1'></i>{{ __('Cancel') }}
                </button>
            </div>
        </form>
    </div>
</div>

<!-- View Task Offcanvas -->
<div class="offcanvas offcanvas-end" tabindex="-1" id="viewTaskOffcanvas" aria-labelledby="viewTaskOffcanvasLabel">
    <div class="offcanvas-header">
        <h5 class="offcanvas-title" id="viewTaskOffcanvasLabel">{{ __('Task Details') }}</h5>
        <button type="button" class="btn-close" data-bs-dismiss="offcanvas" aria-label="Close"></button>
    </div>
    <div class="offcanvas-body">
        <div id="viewTaskContent">
            <!-- Content will be loaded here -->
        </div>
    </div>
</div>
@endsection

@section('script')
    <script src="{{ asset('assets/plugins/notifications/js/lobibox.min.js') }}"></script>
    <script src="{{ asset('assets/plugins/notifications/js/notifications.min.js') }}"></script>
    <script src="{{ asset('assets/plugins/datatable/js/jquery.dataTables.min.js') }}"></script>
    <script src="{{ asset('assets/plugins/datatable/js/dataTables.bootstrap5.min.js') }}"></script>

    <script>
        // Global variables
        var tasksDataCache = tasksDataCache || {};
        
        // Check if mobile device
        function isMobile() {
            return window.innerWidth <= 768;
        }

        // Load tasks data
        function loadTasksData() {
            console.log('Loading tasks data...');
            
            $.ajax({
                url: '{{ route("user.website-visits.index") }}',
                type: 'GET',
                headers: {
                    'X-Requested-With': 'XMLHttpRequest',
                    'Accept': 'application/json'
                },
                success: function(response) {
                    console.log('AJAX Response:', response);
                    
                    if (response.success && response.data) {
                        console.log('Tasks count:', response.data.length);
                        
                        // Debug: Log first task structure
                        if (response.data.length > 0) {
                            console.log('First task structure:', response.data[0]);
                            console.log('Available fields:', Object.keys(response.data[0]));
                        }
                        
                        // Cache all tasks data
                        response.data.forEach(function(task) {
                            tasksDataCache[task.id] = task;
                        });

                        if (isMobile()) {
                            console.log('Rendering mobile view');
                            renderMobileView(response.data);
                        } else {
                            console.log('Rendering desktop table');
                            renderDesktopTable(response.data);
                        }
                    } else {
                        console.error('Invalid response format:', response);
                        $('#tasksContainer').html('<div class="alert alert-warning">{{ __("No data received") }}</div>');
                    }
                },
                error: function(xhr) {
                    console.error('AJAX Error:', xhr);
                    console.error('Response Text:', xhr.responseText);
                    $('#tasksContainer').html('<div class="alert alert-danger">{{ __("Failed to load tasks") }}: ' + xhr.status + '</div>');
                }
            });
        }

        // Desktop DataTable view
        function renderDesktopTable(tasks) {
            if (tasks.length === 0) {
                $('#tasksContainer').html(`
                    <div class="text-center py-5">
                        <i class="bx bx-inbox fs-1 text-muted"></i>
                        <p class="text-muted">{{ __("No tasks found") }}</p>
                        <button type="button" class="btn btn-primary" onclick="openCreateOffcanvas()">
                            <i class='bx bx-plus me-1'></i>{{ __('Add New Task') }}
                        </button>
                    </div>
                `);
                return;
            }

            $('#tasksContainer').html(`
                <div class="table-responsive">
                    <table class="table table-striped" id="tasksTable">
                        <thead>
                            <tr>
                                <th>{{ __('Title') }}</th>
                                <th>{{ __('Website URL') }}</th>
                                <th>{{ __('Reward') }}</th>
                                <th>{{ __('Duration') }}</th>
                                <th>{{ __('Status') }}</th>
                                <th>{{ __('Created') }}</th>
                                <th width="120">{{ __('Actions') }}</th>
                            </tr>
                        </thead>
                        <tbody></tbody>
                    </table>
                </div>
            `);

            console.log('Initializing DataTable with data:', tasks);
            
            const table = $('#tasksTable').DataTable({
                data: tasks,
                destroy: true, // Allow re-initialization
                columns: [
                    { 
                        data: 'title',
                        title: '{{ __("Title") }}',
                        className: 'text-start'
                    },
                    { 
                        data: 'website_url', 
                        title: '{{ __("Website URL") }}',
                        className: 'text-start',
                        render: function(data) {
                            return `<a href="${data}" target="_blank" class="text-primary">${data.length > 30 ? data.substring(0, 30) + '...' : data}</a>`;
                        }
                    },
                    { 
                        data: 'reward_amount', 
                        title: '{{ __("Reward") }}',
                        className: 'text-end',
                        render: function(data) {
                            return parseFloat(data).toFixed(2).replace('.', ',') + '{{ $settings["currency"] ?? "TL" }}';
                        }
                    },
                    { 
                        data: 'visit_duration', 
                        title: '{{ __("Duration") }}',
                        className: 'text-center',
                        render: function(data, type, row) {
                            console.log('Duration data:', data, 'Row:', row);
                            
                            // Check multiple possible field names
                            const duration = data || row.visit_duration || row.required_time || row.duration;
                            
                            if (duration === undefined || duration === null) {
                                console.warn('No duration found in:', row);
                                return '<span class="text-muted">-</span>';
                            }
                            
                            return duration + 's';
                        }
                    },
                    { 
                        data: 'status', 
                        title: '{{ __("Status") }}',
                        className: 'text-center',
                        render: function(data) {
                            return data == 1 ? 
                                '<span class="badge bg-success">{{ __("Active") }}</span>' : 
                                '<span class="badge bg-warning">{{ __("Inactive") }}</span>';
                        }
                    },
                    { 
                        data: 'created_at', 
                        title: '{{ __("Created") }}',
                        className: 'text-center',
                        render: function(data) {
                            return new Date(data).toLocaleDateString('tr-TR');
                        }
                    },
                    { 
                        data: null, // Use null for computed column
                        title: '{{ __("Actions") }}',
                        orderable: false, 
                        searchable: false,
                        className: 'text-center',
                        width: '150px',
                        defaultContent: '', // Prevent undefined errors
                        render: function(data, type, row) {
                            console.log('Rendering actions for task ID:', row.id, 'Row data:', row);
                            
                            if (!row.id) {
                                console.error('No ID found in row:', row);
                                return '<span class="text-muted">No ID</span>';
                            }
                            
                            return `
                                <div class="btn-group btn-group-sm" role="group">
                                    <button type="button" class="btn btn-outline-primary" 
                                            onclick="viewTask(${row.id})" title="{{ __('View Details') }}">
                                        <i class='bx bx-show'></i>
                                    </button>
                                    <button type="button" class="btn btn-outline-warning" 
                                            onclick="editTask(${row.id})" title="{{ __('Edit') }}">
                                        <i class='bx bx-edit'></i>
                                    </button>
                                    <button type="button" class="btn btn-outline-danger" 
                                            onclick="deleteTask(${row.id})" title="{{ __('Delete') }}">
                                        <i class='bx bx-trash'></i>
                                    </button>
                                </div>
                            `;
                        }
                    }
                ],
                order: [[5, 'desc']], // Sort by created date
                pageLength: 15,
                responsive: true,
                autoWidth: false, // Prevent auto width calculation
                columnDefs: [
                    { 
                        targets: -1, // Last column (Actions)
                        width: '150px',
                        orderable: false,
                        searchable: false
                    }
                ],
                language: {
                    processing: '{{ __("Loading...") }}',
                    search: '{{ __("Search") }}:',
                    lengthMenu: '{{ __("Show") }} _MENU_ {{ __("entries") }}',
                    info: '{{ __("Showing") }} _START_ {{ __("to") }} _END_ {{ __("of") }} _TOTAL_ {{ __("entries") }}',
                    infoEmpty: '{{ __("No entries found") }}',
                    infoFiltered: '({{ __("filtered from") }} _MAX_ {{ __("total entries") }})',
                    paginate: {
                        first: '{{ __("First") }}',
                        last: '{{ __("Last") }}',
                        next: '{{ __("Next") }}',
                        previous: '{{ __("Previous") }}'
                    }
                }
            });
            
            console.log('DataTable initialized:', table);
        }

        // Mobile card view
        function renderMobileView(tasks) {
            let html = '<div class="row g-3">';
            
            tasks.forEach(function(task) {
                const statusBadge = task.status == 1 ? 
                    '<span class="badge bg-success">{{ __("Active") }}</span>' : 
                    '<span class="badge bg-warning">{{ __("Inactive") }}</span>';
                
                html += `
                    <div class="col-12">
                        <div class="card task-card" data-task-id="${task.id}">
                            <div class="card-body">
                                <div class="d-flex justify-content-between align-items-start mb-2">
                                    <h6 class="card-title mb-0">${task.title}</h6>
                                    ${statusBadge}
                                </div>
                                
                                <div class="mb-3">
                                    <small class="text-muted d-block">{{ __('Website URL') }}</small>
                                    <a href="${task.website_url}" target="_blank" class="text-primary small">
                                        ${task.website_url.length > 40 ? task.website_url.substring(0, 40) + '...' : task.website_url}
                                    </a>
                                </div>
                                
                                <div class="row g-2 mb-3">
                                    <div class="col-6">
                                        <small class="text-muted d-block">{{ __('Reward') }}</small>
                                        <strong class="text-success">${parseFloat(task.reward_amount).toFixed(2).replace('.', ',')}{{ $settings['currency'] ?? 'TL' }}</strong>
                                    </div>
                                    <div class="col-6">
                                        <small class="text-muted d-block">{{ __('Duration') }}</small>
                                        <span>${task.visit_duration || task.required_time || task.duration || '-'}s</span>
                                    </div>
                                </div>
                                
                                <div class="d-flex gap-2">
                                    <button type="button" class="btn btn-sm btn-outline-primary flex-fill" 
                                            onclick="viewTask(${task.id})">
                                        <i class='bx bx-show me-1'></i>{{ __('View') }}
                                    </button>
                                    <button type="button" class="btn btn-sm btn-outline-warning flex-fill" 
                                            onclick="editTask(${task.id})">
                                        <i class='bx bx-edit me-1'></i>{{ __('Edit') }}
                                    </button>
                                    <button type="button" class="btn btn-sm btn-outline-danger" 
                                            onclick="deleteTask(${task.id})">
                                        <i class='bx bx-trash'></i>
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                `;
            });
            
            html += '</div>';
            
            if (tasks.length === 0) {
                html = `
                    <div class="text-center py-5">
                        <i class="bx bx-inbox fs-1 text-muted"></i>
                        <p class="text-muted">{{ __("No tasks found") }}</p>
                        <button type="button" class="btn btn-primary" onclick="openCreateOffcanvas()">
                            <i class='bx bx-plus me-1'></i>{{ __('Add New Task') }}
                        </button>
                    </div>
                `;
            }
            
            $('#tasksContainer').html(html);
        }

        $(document).ready(function() {
            loadTasksData();
            
            // Handle window resize
            $(window).on('resize', function() {
                clearTimeout(window.resizeTimeout);
                window.resizeTimeout = setTimeout(function() {
                    loadTasksData(); // Reload with appropriate view
                }, 250);
            });
        });

        // Offcanvas Functions
        function openCreateOffcanvas() {
            const offcanvas = new bootstrap.Offcanvas(document.getElementById('createTaskOffcanvas'));
            offcanvas.show();
        }

        function editTask(taskId) {
            const offcanvas = new bootstrap.Offcanvas(document.getElementById('editTaskOffcanvas'));
            offcanvas.show();
            
            // Load countries first
            loadCountries();
            
            // Check if data is already in cache
            if (tasksDataCache[taskId]) {
                populateEditForm(tasksDataCache[taskId]);
                return;
            }
            
            // Load from server if not in cache
            loadTaskData(taskId, function(task) {
                tasksDataCache[taskId] = task;
                populateEditForm(task);
            });
        }

        function populateEditForm(task) {
            console.log('Populating edit form with task:', task);
            
            $('#edit_task_id').val(task.id);
            $('#edit_title').val(task.title);
            $('#edit_description').val(task.description || '');
            $('#edit_website_url').val(task.website_url);
            $('#edit_visit_method').val(task.visit_method || 'direct');
            $('#edit_search_keywords').val(task.search_keywords || '');
            $('#edit_required_time').val(task.required_time);
            $('#edit_reward_amount').val(task.reward_amount);
            $('#edit_max_completions').val(task.max_completions);
            $('#edit_user_daily_limit').val(task.user_daily_limit);
            $('#edit_user_total_limit').val(task.user_total_limit);
            $('#edit_target_gender').val(task.target_gender || '0');
            
            // Handle expires_at
            if (task.expires_at) {
                const date = new Date(task.expires_at);
                const formattedDate = date.toISOString().slice(0, 16);
                $('#edit_expires_at').val(formattedDate);
            } else {
                $('#edit_expires_at').val('');
            }
            
            // Set country and load cities
            const countryCode = task.target_country_id || '0';
            $('#edit_target_country_id').val(countryCode);
            
            // Load cities for selected country and then set the city
            if (countryCode && countryCode !== '0') {
                loadCities(countryCode, function() {
                    $('#edit_target_city_id').val(task.target_city_id || '0');
                });
            } else {
                $('#edit_target_city_id').val('0');
            }
            
            // Toggle search keywords visibility
            toggleEditSearchKeywords();
        }
        
        function toggleEditSearchKeywords() {
            const visitMethod = $('#edit_visit_method').val();
            const searchKeywordsRow = $('#edit_search_keywords_row');
            const searchKeywordsInput = $('#edit_search_keywords');
            
            if (visitMethod === 'google_search') {
                searchKeywordsRow.show();
                searchKeywordsInput.attr('required', true);
            } else {
                searchKeywordsRow.hide();
                searchKeywordsInput.attr('required', false);
            }
        }
        
        // Add event listener for visit method change
        $(document).on('change', '#edit_visit_method', function() {
            toggleEditSearchKeywords();
        });
        
        // Add event listener for country change
        $(document).on('change', '#edit_target_country_id', function() {
            const countryCode = $(this).val();
            loadCities(countryCode);
        });
        
        // Load countries function
        function loadCountries() {
            $.ajax({
                url: '{{ route("user.website-visits.create") }}',
                type: 'GET',
                headers: {
                    'X-Requested-With': 'XMLHttpRequest',
                    'Accept': 'application/json'
                },
                success: function(response) {
                    if (response.countries) {
                        const countrySelect = $('#edit_target_country_id');
                        countrySelect.empty().append('<option value="0">{{ __("All Countries") }}</option>');
                        
                        response.countries.forEach(function(country) {
                            countrySelect.append(`<option value="${country.code}">${country.name}</option>`);
                        });
                    }
                },
                error: function() {
                    console.error('Failed to load countries');
                }
            });
        }
        
        // Load cities function
        function loadCities(countryCode, callback) {
            const citySelect = $('#edit_target_city_id');
            citySelect.empty().append('<option value="0">{{ __("All Cities") }}</option>');
            
            if (countryCode && countryCode !== '0') {
                $.ajax({
                    url: `/cities/${countryCode}`,
                    type: 'GET',
                    success: function(cities) {
                        cities.forEach(function(city) {
                            citySelect.append(`<option value="${city.id}">${city.name}</option>`);
                        });
                        if (callback) callback();
                    },
                    error: function() {
                        console.error('Failed to load cities');
                        if (callback) callback();
                    }
                });
            } else {
                if (callback) callback();
            }
        }

        function loadTaskData(taskId, callback) {
            $.ajax({
                url: `{{ url('/my/website-visits') }}/${taskId}`,
                type: 'GET',
                headers: {
                    'X-Requested-With': 'XMLHttpRequest',
                    'Accept': 'application/json'
                },
                success: function(response) {
                    if (response.success) {
                        callback(response.data);
                    } else {
                        Swal.fire('{{ __("Error") }}', response.message, 'error');
                    }
                },
                error: function(xhr) {
                    const message = xhr.responseJSON?.message || '{{ __("Failed to load task data") }}';
                    Swal.fire('{{ __("Error") }}', message, 'error');
                }
            });
        }

        // Global tasks data cache

        function viewTask(taskId) {
            const offcanvas = new bootstrap.Offcanvas(document.getElementById('viewTaskOffcanvas'));
            offcanvas.show();
            
            // Check if data is already in cache (from DataTable)
            if (tasksDataCache[taskId]) {
                const content = generateTaskViewContent(tasksDataCache[taskId]);
                $('#viewTaskContent').html(content);
                return;
            }
            
            // Show loading only if not in cache
            $('#viewTaskContent').html('<div class="text-center"><i class="bx bx-loader-alt bx-spin fs-3"></i><p>{{ __("Loading...") }}</p></div>');
            
            // Load from server and cache
            loadTaskData(taskId, function(task) {
                tasksDataCache[taskId] = task;
                const content = generateTaskViewContent(task);
                $('#viewTaskContent').html(content);
            });
        }

        function generateTaskViewContent(task) {
            const statusBadge = task.status == 1 ? 
                '<span class="badge bg-success">{{ __("Active") }}</span>' : 
                '<span class="badge bg-warning">{{ __("Inactive") }}</span>';
                
            return `
                <div class="row g-3">
                    <div class="col-12">
                        <div class="card">
                            <div class="card-body">
                                <h6 class="card-title">${task.title}</h6>
                                <p class="card-text">${task.description || '{{ __("No description") }}'}</p>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-12">
                        <div class="list-group">
                            <div class="list-group-item d-flex justify-content-between">
                                <strong>{{ __("Website URL") }}:</strong>
                                <a href="${task.website_url}" target="_blank" class="text-primary">${task.website_url}</a>
                            </div>
                            <div class="list-group-item d-flex justify-content-between">
                                <strong>{{ __("Reward Amount") }}:</strong>
                                <span>${parseFloat(task.reward_amount).toFixed(2)}{{ $settings['currency'] ?? 'TL' }}</span>
                            </div>
                            <div class="list-group-item d-flex justify-content-between">
                                <strong>{{ __("Visit Duration") }}:</strong>
                                <span>${task.visit_duration} {{ __("seconds") }}</span>
                            </div>
                            <div class="list-group-item d-flex justify-content-between">
                                <strong>{{ __("Daily Limit") }}:</strong>
                                <span>${task.user_daily_limit || '{{ __("Unlimited") }}'}</span>
                            </div>
                            <div class="list-group-item d-flex justify-content-between">
                                <strong>{{ __("Status") }}:</strong>
                                ${statusBadge}
                            </div>
                            <div class="list-group-item d-flex justify-content-between">
                                <strong>{{ __("Created") }}:</strong>
                                <span>${new Date(task.created_at).toLocaleDateString()}</span>
                            </div>
                        </div>
                    </div>
                </div>
            `;
        }

        // Form submission functions
        function submitCreateForm() {
            const form = document.getElementById('createTaskForm');
            const formData = new FormData(form);
            const button = document.getElementById('createTaskBtn');
            const originalHtml = button.innerHTML;

            // Update button state
            button.disabled = true;
            button.innerHTML = '<i class="bx bx-loader-alt bx-spin me-1"></i>{{ __("Creating...") }}';

            $.ajax({
                url: '{{ route("user.website-visits.store") }}',
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                headers: {
                    'X-Requested-With': 'XMLHttpRequest'
                },
                success: function(response) {
                    if (response.success) {
                        // Close offcanvas
                        const offcanvas = bootstrap.Offcanvas.getInstance(document.getElementById('createTaskOffcanvas'));
                        offcanvas.hide();

                        // Reset form
                        form.reset();

                        // Show success notification with Lobibox
                        showNotification('success', '{{ __("Success") }}', response.message);

                        // Reload tasks data
                        setTimeout(() => {
                            loadTasksData();
                        }, 1000);
                    } else {
                        showNotification('error', '{{ __("Error") }}', response.message);
                    }
                },
                error: function(xhr) {
                    let message = '{{ __("An error occurred") }}';
                    if (xhr.responseJSON) {
                        if (xhr.responseJSON.errors) {
                            const errors = Object.values(xhr.responseJSON.errors).flat();
                            message = errors.join('<br>');
                        } else if (xhr.responseJSON.message) {
                            message = xhr.responseJSON.message;
                        }
                    }
                    showNotification('error', '{{ __("Error") }}', message);
                },
                complete: function() {
                    // Restore button state
                    button.disabled = false;
                    button.innerHTML = originalHtml;
                }
            });
        }

        function submitEditForm() {
            const form = document.getElementById('editTaskForm');
            const formData = new FormData(form);
            const taskId = document.getElementById('edit_task_id').value;
            const button = document.getElementById('editTaskBtn');
            const originalHtml = button.innerHTML;

            // Add _method=POST for Laravel
            formData.append('_method', 'POST');

            // Update button state
            button.disabled = true;
            button.innerHTML = '<i class="bx bx-loader-alt bx-spin me-1"></i>{{ __("Updating...") }}';

            $.ajax({
                url: `{{ url('/my/website-visits') }}/${taskId}`,
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                headers: {
                    'X-Requested-With': 'XMLHttpRequest',
                    'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
                },
                success: function(response) {
                    if (response.success) {
                        // Close offcanvas
                        const offcanvas = bootstrap.Offcanvas.getInstance(document.getElementById('editTaskOffcanvas'));
                        offcanvas.hide();

                        // Show success notification with Lobibox
                        showNotification('success', '{{ __("Success") }}', response.message);

                        // Reload tasks data
                        setTimeout(() => {
                            loadTasksData();
                        }, 1000);
                    } else {
                        showNotification('error', '{{ __("Error") }}', response.message);
                    }
                },
                error: function(xhr) {
                    let message = '{{ __("An error occurred") }}';
                    if (xhr.responseJSON) {
                        if (xhr.responseJSON.errors) {
                            const errors = Object.values(xhr.responseJSON.errors).flat();
                            message = errors.join('<br>');
                        } else if (xhr.responseJSON.message) {
                            message = xhr.responseJSON.message;
                        }
                    }
                    showNotification('error', '{{ __("Error") }}', message);
                },
                complete: function() {
                    // Restore button state
                    button.disabled = false;
                    button.innerHTML = originalHtml;
                }
            });
        }

        // Form event listeners
        $(document).ready(function() {
            $('#createTaskForm').on('submit', function(e) {
                e.preventDefault();
                submitCreateForm();
            });

            $('#editTaskForm').on('submit', function(e) {
                e.preventDefault();
                submitEditForm();
            });
        });

        function deleteTask(taskId) {
            Swal.fire({
                title: '{{ __("Are you sure?") }}',
                text: '{{ __("This action cannot be undone! All related data will be permanently deleted.") }}',
                icon: 'warning',
                showCancelButton: true,
                confirmButtonColor: '#d33',
                cancelButtonColor: '#6c757d',
                confirmButtonText: '{{ __("Yes, delete it!") }}',
                cancelButtonText: '{{ __("Cancel") }}',
                reverseButtons: true
            }).then((result) => {
                if (result.isConfirmed) {
                    // Show loading with SweetAlert
                    Swal.fire({
                        title: '{{ __("Deleting...") }}',
                        text: '{{ __("Please wait while we delete the task.") }}',
                        icon: 'info',
                        allowOutsideClick: false,
                        showConfirmButton: false,
                        willOpen: () => {
                            Swal.showLoading();
                        }
                    });

                    $.ajax({
                        url: `{{ url('/my/website-visits') }}/${taskId}`,
                        type: 'POST',
                        data: {
                            '_method': 'DELETE'
                        },
                        headers: {
                            'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content'),
                            'X-Requested-With': 'XMLHttpRequest'
                        },
                        success: function(response) {
                            if (response.success) {
                                Swal.fire({
                                    title: '{{ __("Deleted!") }}',
                                    text: response.message,
                                    icon: 'success',
                                    timer: 2000,
                                    showConfirmButton: false
                                });
                                
                                // Reload tasks data
                                setTimeout(() => {
                                    loadTasksData();
                                }, 1000);
                            } else {
                                Swal.fire({
                                    title: '{{ __("Error") }}',
                                    text: response.message,
                                    icon: 'error'
                                });
                            }
                        },
                        error: function(xhr) {
                            const message = xhr.responseJSON?.message || '{{ __("An error occurred") }}';
                            Swal.fire({
                                title: '{{ __("Error") }}',
                                text: message,
                                icon: 'error'
                            });
                        }
                    });
                }
            });
        }

        function showNotification(type, title, message) {
            Lobibox.notify(type, {
                pauseDelayOnHover: true,
                continueDelayOnInactiveTab: false,
                position: 'top right',
                icon: type === 'success' ? 'bx bx-check-circle' : 'bx bx-x-circle',
                title: title,
                msg: message
            });
        }
    </script>
@endsection