@extends('layouts.app')

@section('style')
<style>
    .watch-container {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        border-radius: 20px;
        padding: 30px;
        color: white;
        margin-bottom: 30px;
    }
    
    .task-card {
        background: white;
        border-radius: 15px;
        transition: transform 0.3s ease, box-shadow 0.3s ease;
        overflow: hidden;
        border: none;
        box-shadow: 0 5px 15px rgba(0,0,0,0.1);
    }
    
    .task-card:hover {
        transform: translateY(-5px);
        box-shadow: 0 15px 35px rgba(0,0,0,0.2);
    }
    
    .task-thumbnail {
        height: 200px;
        background-size: cover;
        background-position: center;
        position: relative;
        overflow: hidden;
    }
    
    .task-thumbnail::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: linear-gradient(45deg, rgba(102, 126, 234, 0.8), rgba(118, 75, 162, 0.8));
        opacity: 0;
        transition: opacity 0.3s ease;
    }
    
    .task-card:hover .task-thumbnail::before {
        opacity: 1;
    }
    
    .play-button {
        position: absolute;
        top: 50%;
        left: 50%;
        transform: translate(-50%, -50%);
        background: rgba(255,255,255,0.9);
        border-radius: 50%;
        width: 60px;
        height: 60px;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 24px;
        color: #667eea;
        opacity: 0;
        transition: opacity 0.3s ease;
    }
    
    .task-card:hover .play-button {
        opacity: 1;
    }
    
    .reward-badge {
        position: absolute;
        top: 15px;
        right: 15px;
        background: linear-gradient(45deg, #28a745, #20c997);
        color: white;
        padding: 5px 12px;
        border-radius: 20px;
        font-weight: bold;
        font-size: 14px;
    }
    
    .duration-badge {
        position: absolute;
        bottom: 15px;
        left: 15px;
        background: rgba(0,0,0,0.7);
        color: white;
        padding: 4px 8px;
        border-radius: 10px;
        font-size: 12px;
    }
    
    .task-status {
        display: inline-block;
        padding: 4px 12px;
        border-radius: 15px;
        font-size: 12px;
        font-weight: bold;
    }
    
    .status-completed {
        background: #d4edda;
        color: #155724;
    }
    
    .status-available {
        background: #d1ecf1;
        color: #0c5460;
    }
    
    .status-unavailable {
        background: #f8d7da;
        color: #721c24;
    }
    
    .add-task-btn {
        position: fixed;
        bottom: 30px;
        right: 30px;
        width: 60px;
        height: 60px;
        border-radius: 50%;
        background: linear-gradient(45deg, #667eea, #764ba2);
        border: none;
        color: white;
        font-size: 24px;
        box-shadow: 0 10px 25px rgba(102, 126, 234, 0.3);
        transition: transform 0.3s ease;
        z-index: 1000;
    }
    
    .add-task-btn:hover {
        transform: scale(1.1);
        color: white;
    }
    
    .offcanvas-custom {
        width: 100% !important;
        max-width: 500px;
    }
    
    @media (max-width: 768px) {
        .watch-container {
            padding: 20px;
            margin: 10px;
            border-radius: 15px;
        }
        
        .task-thumbnail {
            height: 150px;
        }
        
        .add-task-btn {
            bottom: 20px;
            right: 20px;
            width: 50px;
            height: 50px;
            font-size: 20px;
        }
        
        .offcanvas-custom {
            width: 100% !important;
        }
    }
    
    .form-control:focus {
        border-color: #667eea;
        box-shadow: 0 0 0 0.2rem rgba(102, 126, 234, 0.25);
    }
    
    .btn-primary {
        background: linear-gradient(45deg, #667eea, #764ba2);
        border: none;
    }
    
    .btn-primary:hover {
        background: linear-gradient(45deg, #5a67d8, #6b46c1);
        transform: translateY(-1px);
    }

      .balance-card {
        background: linear-gradient(45deg, #28a745, #20c997);
        color: white;
        border-radius: 15px;
        padding: 25px;
        text-align: center;
        margin-bottom: 30px;
    }
    
    .balance-amount {
        font-size: 2.5rem;
        font-weight: bold;
        margin-bottom: 10px;
    }
</style>
@endsection

@section('wrapper')
<div class="page-wrapper">
    <div class="page-content">
        <!-- Header Section -->
        <div class="watch-container">
            <div class="row align-items-center">
                <div class="col-md-8">
                    <h2 class="mb-3 text-white">
                        <i class="bx bx-play-circle me-2"></i>
                        {{__('Watch Videos & Earn Money')}}
                    </h2>
                    <p class="mb-0 opacity-90">
                        {{__('Watch YouTube videos and shorts to earn rewards. Complete the required watch time to claim your earnings!')}}
                    </p>
                </div>
                <div class="col-md-4 text-end">
                    <div class="d-flex flex-column align-items-end">
                           <div class="balance-card">
                        <div class="balance-amount">{{ number_format($user->balance, 2) }}{{ $currency }}</div>
                        <small class="opacity-75">{{__('Your Balance')}}</small>
                    </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Tasks Grid -->
        <div class="row" id="tasks-container">
            <!-- Tasks will be loaded here via AJAX -->
        </div>

        <!-- Empty State -->
        <div class="row d-none" id="empty-state">
            <div class="col-12">
                <div class="card task-card">
                    <div class="card-body text-center py-5">
                        <i class="bx bx-video" style="font-size: 4rem; color: #ddd;"></i>
                        <h5 class="mt-3 text-muted">{{__('No watch tasks available')}}</h5>
                        <p class="text-muted">{{__('Check back later for new videos to watch and earn!')}}</p>
                        <button class="btn btn-primary" onclick="showAddTaskModal()">
                            <i class="bx bx-plus me-2"></i>{{__('Create First Task')}}
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Add Task Button -->
<button class="btn add-task-btn" onclick="showAddTaskModal()" title="{{__('Add New Watch Task')}}">
    <i class="bx bx-plus"></i>
</button>

<!-- Add Task Offcanvas -->
<div class="offcanvas offcanvas-end offcanvas-custom" tabindex="-1" id="addTaskOffcanvas">
    <div class="offcanvas-header">
        <h5 class="offcanvas-title">
            <i class="bx bx-plus-circle me-2"></i>
            {{__('Add New Watch Task')}}
        </h5>
        <button type="button" class="btn-close" data-bs-dismiss="offcanvas"></button>
    </div>
    <div class="offcanvas-body">
        <form id="addTaskForm">
            @csrf
            <div class="mb-3">
                <label class="form-label">{{__('Task Title')}}</label>
                <input type="text" class="form-control" name="title" required>
                <div class="invalid-feedback"></div>
            </div>
            
            <div class="mb-3">
                <label class="form-label">{{__('Description')}} ({{__('Optional')}})</label>
                <textarea class="form-control" name="description" rows="3"></textarea>
                <div class="invalid-feedback"></div>
            </div>
            
            <div class="mb-3">
                <label class="form-label">{{__('YouTube URL')}}</label>
                <input type="url" class="form-control" name="youtube_url" required 
                       placeholder="https://www.youtube.com/watch?v=...">
                <div class="invalid-feedback"></div>
                <small class="text-muted">{{__('YouTube video or shorts URL')}}</small>
            </div>
            
            <div class="row">
                <div class="col-6">
                    <div class="mb-3">
                        <label class="form-label">{{__('Watch Duration')}} ({{__('seconds')}})</label>
                        <input type="number" class="form-control" name="watch_duration" 
                               min="10" max="3600" required>
                        <div class="invalid-feedback"></div>
                        <small class="text-muted">10-3600 {{__('seconds')}}</small>
                    </div>
                </div>
                <div class="col-6">
                    <div class="mb-3">
                        <label class="form-label">{{__('Reward Amount')}} (₺)</label>
                        <input type="number" class="form-control" name="reward_amount" 
                               min="0.01" max="1000" step="0.01" required>
                        <div class="invalid-feedback"></div>
                    </div>
                </div>
            </div>
            
            <div class="mb-3">
                <label class="form-label">{{__('Max Completions')}}</label>
                <input type="number" class="form-control" name="max_completions" 
                       min="1" max="10000" required>
                <div class="invalid-feedback"></div>
                <small class="text-muted">{{__('How many users can complete this task')}}</small>
            </div>
            
            <div class="mb-3">
                <div class="alert alert-info">
                    <h6 class="mb-2">{{__('Cost Calculation')}}</h6>
                    <div class="d-flex justify-content-between">
                        <span>{{__('Reward per completion')}}:</span>
                        <span id="reward-preview">0.00₺</span>
                    </div>
                    <div class="d-flex justify-content-between">
                        <span>{{__('Max completions')}}:</span>
                        <span id="completions-preview">0</span>
                    </div>
                    <hr>
                    <div class="d-flex justify-content-between fw-bold">
                        <span>{{__('Total Cost')}}:</span>
                        <span id="total-cost-preview">0.00₺</span>
                    </div>
                    <small class="text-muted d-block mt-2">
                        {{__('Your Balance')}}: {{ number_format($user->balance, 2) }}₺
                    </small>
                </div>
            </div>
            
            <div class="d-grid gap-2">
                <button type="submit" class="btn btn-primary" id="submitBtn">
                    <i class="bx bx-check me-2"></i>{{__('Create Task')}}
                </button>
                <button type="button" class="btn btn-secondary" data-bs-dismiss="offcanvas">
                    {{__('Cancel')}}
                </button>
            </div>
        </form>
    </div>
</div>
@endsection

@section('script')
<script>
let userBalance = {{ $user->balance }};

$(document).ready(function() {
    loadTasks();
    $('input[name="reward_amount"], input[name="max_completions"]').on('input', updateCostPreview);
});

function loadTasks() {
    $.get('{{ route("api.watch.tasks") }}', function(response) {
        if (response.success) {
            displayTasks(response.tasks);
        }
    });
}

function displayTasks(tasks) {
    const container = $('#tasks-container');
    const emptyState = $('#empty-state');

    if (tasks.length === 0) {
        container.empty();
        emptyState.removeClass('d-none');
        return;
    }

    emptyState.addClass('d-none');
    container.empty();

    tasks.forEach(task => {
        const taskCard = createTaskCard(task);
        container.append(taskCard);
    });
}

function createTaskCard(task) {
    const thumbnailUrl = `https://img.youtube.com/vi/${task.youtube_url.split('v=')[1]?.split('&')[0] || task.youtube_url.split('/').pop()}/maxresdefault.jpg`;

    let statusClass = 'status-unavailable';
    let statusText = '{{__("Unavailable")}}';

    if (task.is_completed) {
        statusClass = 'status-completed';
        statusText = '{{__("Completed")}}';
    } else if (task.can_watch) {
        statusClass = 'status-available';
        statusText = '{{__("Available")}}';
    }

    return `
        <div class="col-lg-4 col-md-6 mb-4">
            <div class="card task-card">
                <div class="task-thumbnail" style="background-image: url('${thumbnailUrl}')">
                    <div class="reward-badge">${task.reward_amount}₺</div>
                    <div class="duration-badge">${task.formatted_duration}</div>
                    ${task.can_watch ? `
                        <div class="play-button">
                            <i class="bx bx-play"></i>
                        </div>
                    ` : ''}
                </div>
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-start mb-2">
                        <h6 class="card-title mb-0">${task.title}</h6>
                        <span class="task-status ${statusClass}">${statusText}</span>
                    </div>
                    ${task.description ? `<p class="card-text text-muted small">${task.description}</p>` : ''}
                    <div class="row text-center mb-3">
                        <div class="col-4">
                            <small class="text-muted d-block">{{__('Progress')}}</small>
                            <strong>${task.current_completions}/${task.max_completions}</strong>
                        </div>
                        <div class="col-4">
                            <small class="text-muted d-block">{{__('Duration')}}</small>
                            <strong>${task.formatted_duration}</strong>
                        </div>
                        <div class="col-4">
                            <small class="text-muted d-block">{{__('Creator')}}</small>
                            <strong>${task.created_by}</strong>
                        </div>
                    </div>
                    <div class="d-grid gap-2">
                        ${task.can_watch ? `
                           <a href="/watch/${task.id}" class="btn btn-primary btn-sm">
                                <i class="bx bx-play me-1"></i>{{__('Watch Now')}}
                            </a>
                        ` : task.is_completed ? `
                            <button class="btn btn-success btn-sm" disabled>
                                <i class="bx bx-check me-1"></i>{{__('Completed')}}
                            </button>
                        ` : `
                            <button class="btn btn-secondary btn-sm" disabled>
                                {{__('Not Available')}}
                            </button>
                        `}
                        ${task.can_manage ? `
                            <div class="btn-group" role="group">
                                <button class="btn btn-outline-warning btn-sm" onclick="toggleTaskStatus(${task.id}, ${task.is_active})">
                                    <i class="bx ${task.is_active ? 'bx-pause' : 'bx-play'}"></i>
                                    ${task.is_active ? '{{__("Deactivate")}}' : '{{__("Activate")}}'}
                                </button>
                                <button class="btn btn-outline-danger btn-sm" onclick="deleteTask(${task.id})">
                                    <i class="bx bx-trash"></i>{{__('Delete')}}
                                </button>
                            </div>
                        ` : ''}
                    </div>
                </div>
            </div>
        </div>
    `;
}

function showAddTaskModal() {
    const offcanvas = new bootstrap.Offcanvas(document.getElementById('addTaskOffcanvas'));
    offcanvas.show();
}

function updateCostPreview() {
    const reward = parseFloat($('input[name="reward_amount"]').val()) || 0;
    const completions = parseInt($('input[name="max_completions"]').val()) || 0;
    const totalCost = reward * completions;
    
    $('#reward-preview').text(reward.toFixed(2) + '₺');
    $('#completions-preview').text(completions);
    $('#total-cost-preview').text(totalCost.toFixed(2) + '₺');
    
    // Check if user has enough balance
    const submitBtn = $('#submitBtn');
    if (totalCost > userBalance) {
        submitBtn.prop('disabled', true).html('<i class="bx bx-x me-2"></i>{{__("Insufficient Balance")}}');
    } else {
        submitBtn.prop('disabled', false).html('<i class="bx bx-check me-2"></i>{{__("Create Task")}}');
    }
}

$('#addTaskForm').on('submit', function(e) {
    e.preventDefault();
    
    const submitBtn = $('#submitBtn');
    const originalText = submitBtn.html();
    
    submitBtn.prop('disabled', true).html('<i class="bx bx-loader-alt bx-spin me-2"></i>{{__("Creating...")}}');
    
    // Clear previous errors
    $('.is-invalid').removeClass('is-invalid');
    $('.invalid-feedback').empty();
    
    $.ajax({
        url: '{{ route("api.watch.store") }}',
        method: 'POST',
        data: $(this).serialize(),
        success: function(response) {
            if (response.success) {
                // Update user balance
                userBalance = response.new_balance;
                
                // Close offcanvas
                bootstrap.Offcanvas.getInstance(document.getElementById('addTaskOffcanvas')).hide();
                
                // Reset form
                $('#addTaskForm')[0].reset();
                updateCostPreview();
                
                // Reload tasks
                loadTasks();
                
                // Show success message
                Swal.fire({
                    icon: 'success',
                    title: '{{__("Success")}}',
                    text: response.message,
                    timer: 3000
                });
                
                // Update balance display
                $('.watch-container h4').text(response.new_balance.toFixed(2) + '₺');
            }
        },
        error: function(xhr) {
            const errors = xhr.responseJSON?.errors || {};
            
            // Display validation errors
            Object.keys(errors).forEach(field => {
                const input = $(`[name="${field}"]`);
                input.addClass('is-invalid');
                input.siblings('.invalid-feedback').text(errors[field][0]);
            });
            
            // Show general error message
            if (xhr.responseJSON?.message) {
                Swal.fire({
                    icon: 'error',
                    title: '{{__("Error")}}',
                    text: xhr.responseJSON.message
                });
            }
        },
        complete: function() {
            submitBtn.prop('disabled', false).html(originalText);
        }
    });
});

function toggleTaskStatus(taskId) {
  $.ajax({
    url: `/api/watch-tasks/${taskId}/toggle`,
    method: 'POST',
    headers: {
      'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content'),
      'Accept':       'application/json'
    },
    success: function(response) {
      // Burada loadMyTasks() yerine loadTasks() çağrılıyor:
      loadTasks();
      Swal.fire({
        icon: 'success',
        text: response.message,
        timer: 1500
      });
    },
    error: function(xhr) {
      Swal.fire({
        icon: 'error',
        text: xhr.responseJSON?.message || 'Bir hata oluştu.'
      });
    }
  });
}


function deleteTask(taskId) {
    Swal.fire({
        title: '{{__("Are you sure?")}}',
        text: '{{__("This action cannot be undone! All related data will be permanently deleted.")}}',
        icon: 'warning',
        showCancelButton: true,
        confirmButtonColor: '#d33',
        cancelButtonColor: '#3085d6',
        confirmButtonText: '{{__("Yes, delete it!")}}',
        cancelButtonText: '{{__("Cancel")}}'
    }).then((result) => {
        if (result.isConfirmed) {
            $.ajax({
                url: `/api/watch-tasks/${taskId}`,
                method: 'DELETE',
                data: {
                    _token: $('meta[name="csrf-token"]').attr('content')
                },
                success: function(response) {
                    if (response.success) {
                        loadTasks();

                        if (response.refund_amount > 0) {
                            userBalance += response.refund_amount;
                            $('.watch-container h4').text(userBalance.toFixed(2) + '₺');
                        }

                        Swal.fire({
                            icon: 'success',
                            title: '{{__("Deleted!")}}',
                            text: response.message,
                            timer: 3000
                        });
                    }
                },
                error: function(xhr) {
                    Swal.fire({
                        icon: 'error',
                        title: '{{__("Error")}}',
                        text: xhr.responseJSON?.message || '{{__("An error occurred")}}'
                    });
                }
            });
        }
    });
}
</script>
@endsection